package me.yoann.tp3;

import java.util.HashSet;

import org.graphstream.algorithm.APSP;
import org.graphstream.algorithm.APSP.APSPInfo;
import org.graphstream.algorithm.Dijkstra;
import org.graphstream.algorithm.Toolkit;
import org.graphstream.graph.BreadthFirstIterator;
import org.graphstream.graph.Edge;
import org.graphstream.graph.Graph;
import org.graphstream.graph.Node;
import org.graphstream.graph.Path;
import org.graphstream.graph.implementations.SingleGraph;
import org.graphstream.stream.file.FileSourceDGS;
import org.graphstream.ui.view.Viewer;
import org.graphstream.ui.view.ViewerListener;
import org.graphstream.ui.view.ViewerPipe;

import me.yoann.tp3.utils.IOUtils;

/**
 * Classe répondant aux besoins du TP.
 * 
 * @author Yoann Eichelberger 14/12/18
 */
public class App {

    private Graph graph;
    private boolean loop;

    public App(Graph graph) {
        this.graph = graph;
        this.loop = false;
    }

    public void start() {
        Viewer viewer = graph.display(false);
        IOUtils.readFile("lh.dgs", graph, new FileSourceDGS());
        graph.addAttribute("ui.stylesheet", "url('lh.css')");
        viewer.setCloseFramePolicy(Viewer.CloseFramePolicy.EXIT);
        ViewerPipe fromViewer = viewer.newViewerPipe();
        fromViewer.addViewerListener(new GPSViewerListener());
        fromViewer.addSink(graph);

        System.out.printf("Nombre de sommets         : %d%n", graph.getNodeCount());
        System.out.printf("Nombre d'arrêtes          : %d%n", graph.getEdgeCount());
        System.out.printf("Degré moyen               : %f%n", Toolkit.averageDegree(graph));
        System.out.printf("Coefficient de clustering : %f%n", Toolkit.averageClusteringCoefficient(graph));
        System.out.printf("Connexe                   : %s%n", Toolkit.isConnected(graph));

        int nodeCount = graph.getNodeCount();
        IOUtils.writeDistribution("data/ddegree_lh.dat", Toolkit.degreeDistribution(graph),
                v -> (double) v / nodeCount);

        long begin, end;
        double diameter;

        begin = System.currentTimeMillis();
        diameter = Toolkit.diameter(graph, "length", false);
        end = System.currentTimeMillis();
        System.out.printf("Diamètre (Toolkit)        : %.5f (%.5f s)%n", diameter, (end - begin) / 1000D);

        begin = System.currentTimeMillis();
        diameter = dijkstraDiameter(graph);
        end = System.currentTimeMillis();
        System.out.printf("Diamètre (Dijkstra)       : %.5f (%.5f s)%n", diameter, (end - begin) / 1000D);

        begin = System.currentTimeMillis();
        diameter = floydWarshallDiameter(graph);
        end = System.currentTimeMillis();
        System.out.printf("Diamètre (Floyd-Warshall) : %.5f (%.5f s)%n", diameter, (end - begin) / 1000D);

        this.loop = true;
        while (this.loop) {
            fromViewer.pump();
        }
    }

    public double floydWarshallDiameter(Graph g) {
        APSP apsp = new APSP();
        apsp.init(graph);
        apsp.setDirected(false);
        apsp.setWeightAttributeName("length");
        apsp.compute();
        double lMax = 0;
        for (Node n : g) {
            APSPInfo info = n.getAttribute(APSPInfo.ATTRIBUTE_NAME);
            if (info != null) {
                BreadthFirstIterator<Node> bfi = new BreadthFirstIterator<>(n);
                while (bfi.hasNext()) {
                    double l = info.getLengthTo(bfi.next().getId());
                    if (l > lMax)
                        lMax = l;
                }
            }
        }
        return lMax;
    }

    public double dijkstraDiameter(Graph g) {
        Dijkstra dijkstra = new Dijkstra(Dijkstra.Element.EDGE, "result", "length");
        dijkstra.init(graph);
        double lMax = 0;
        for (Node n : g) {
            dijkstra.setSource(n);
            dijkstra.compute();
            BreadthFirstIterator<Node> bfi = new BreadthFirstIterator<>(n);
            while (bfi.hasNext()) {
                Node u = bfi.next();
                double l = dijkstra.getPathLength(u);
                if (l > lMax)
                    lMax = l;
            }
        }
        return lMax;
    }

    public Path route(Node a, Node b) {
        Dijkstra dijkstra = new Dijkstra(Dijkstra.Element.EDGE, "result", "length");
        dijkstra.init(graph);
        dijkstra.setSource(a);
        dijkstra.compute();

        return dijkstra.getPath(b);
    }

    public static void main(String[] args) throws Exception {
        App app = new App(new SingleGraph("ville"));
        app.start();
    }

    private class GPSViewerListener implements ViewerListener {

        private Node departure;
        private Node destination;
        private Iterable<Edge> edgePath = new HashSet<>();

        public void setDeparture(Node departure) {
            this.departure = departure;
        }

        public void setDestination(Node destination) {
            this.destination = destination;
        }

        @Override
        public void buttonPushed(String id) {
            System.out.println("Node : " + id);
            Graph g = App.this.graph;
            Node n = g.getNode(id);
            if (this.departure == null) {
                this.departure = n;
                this.departure.addAttribute("ui.class", "departure");
            } else if (this.destination == null) {
                this.destination = n;
                this.destination.addAttribute("ui.class", "destination");
                this.edgePath = route(this.departure, this.destination).getEdgePath();
                for (Edge edge : this.edgePath) {
                    edge.addAttribute("ui.class", "path");
                }
            } else {
                this.departure.removeAttribute("ui.class");
                this.destination.removeAttribute("ui.class");
                for (Edge edge : edgePath) {
                    edge.removeAttribute("ui.class");
                }
                this.departure = null;
                this.destination = null;
            }
        }

        @Override
        public void buttonReleased(String id) {
        }

        @Override
        public void viewClosed(String id) {
            App.this.loop = false;
        }
    }
}