package ai.queens;

import static java.lang.Math.floor;
import static java.lang.Math.random;

import java.util.Arrays;

import ai.Resolver;
import ai.metaheuristic.genetic.GeneticAlgorithm;
import ai.metaheuristic.genetic.Individual;
import ai.metaheuristic.genetic.selector.BiasedWheelOperator;

/**
 * @author Yoann Eichelberger - M1 IWOCS
 */
public class QueenResolver implements Resolver<int[]> {

    private GeneticAlgorithm<QueenIndividual> genetic;

    /**
     * Construit une instance d'un resolver d'un problème à N-Reines.
     * 
     * @param size nombre de reines.
     */
    public QueenResolver(int size) {
        this.genetic = new GeneticAlgorithm<QueenIndividual>(5000, QueenIndividual[]::new,
                new BiasedWheelOperator<>(fit -> fit + 0.1), 0.15f, 0.05f, 0.01f, 0.00f) {
            @Override
            public QueenIndividual generate() {
                return new QueenIndividual(size);
            }

            @Override
            public QueenIndividual mutate(QueenIndividual individual) {
                int n = individual.queens.length;
                if (n <= 1) {
                    return individual;
                }
                int a = (int) (floor(random() * n));
                int b = (int) (a + 1 + floor(random() * n)) % n;
                int[] copy = Arrays.copyOf(individual.queens, n);
                int tmp = copy[b];
                copy[b] = copy[a];
                copy[a] = tmp;
                return new QueenIndividual(copy);
            }

            @Override
            public QueenIndividual crossover(QueenIndividual i1, QueenIndividual i2) {
                int n = size;
                if (i2.lines == null) {
                    i2.lines = new int[n];
                    for (int i = 0; i < n; i++) {
                        i2.lines[i2.queens[i]] = i;
                    }
                }
                int[] nq = Arrays.copyOf(i2.queens, n);
                for (int z = 0; z < 1; z++) {
                    int k1 = (int) Math.floor(random() * n);
                    int k2 = (int) Math.floor(random() * n);
                    int b = Math.min(k1, k2);
                    int e = Math.max(k1, k2);
                    int[] lines = new int[e - b];
                    for (int i = 0; i < e - b; i++) {
                        lines[i] = i2.lines[i1.queens[b + i]];
                    }
                    Arrays.sort(lines);
                    for (int i = 0; i < e - b; i++) {
                        nq[lines[i]] = i1.queens[b + i];
                    }
                }
                return new QueenIndividual(nq);
            }
        };
    }

    /**
     * Recherche une solution.
     * 
     * @return le tableau de reines répondant au problème.
     */
    @Override
    public int[] search() {
        return this.genetic.geneticAlgorithm().queens;
    }

    /**
     * Représente une configuration du placement des reines.
     */
    private static class QueenIndividual extends Individual {

        private int[] queens;
        private int[] lines;
        private double fitness;

        /**
         * Construit un échiquier et place des reines aléatoirement.
         */
        private QueenIndividual(int size) {
            this(QueenIndividual.build(size));
        }

        /**
         * Construit un échiquier à partir d'un autre échiquier.
         */
        private QueenIndividual(int[] queens) {
            this.queens = queens;
            this.lines = null;
            this.fitness = -1;
        }

        /**
         * 
         * @param l
         * @param c
         * @return
         */
        private boolean isTaken(int l, int c) {
            for (int j = 0; j < l; j++) {
                if (this.queens[j] == c || l - j == c - this.queens[j] || l - j == this.queens[j] - c) {
                    return true;
                }
            }
            return false;
        }

        /**
         * {@inheritdoc}
         */
        @Override
        public int length() {
            return this.queens.length;
        }

        /**
         * 
         */
        @Override
        public double fitness() {
            if (this.fitness == -1) {
                this.fitness = 0;
                for (int i = 1; i < queens.length; i++) {
                    if (!this.isTaken(i, queens[i])) {
                        this.fitness++;
                    }
                }
                this.fitness /= queens.length - 1;
            }
            return this.fitness;
        }

        /**
         * Construit un échiquier aléatoirement.
         */
        private static int[] build(int size) {
            int lastIndex = size - 1;
            int[] bag = new int[size];
            for (int i = 0; i < size; i++) {
                bag[i] = i;
            }
            int[] queens = new int[size];
            for (int i = 0; i < size; i++) {
                int j = (int) (random() * lastIndex);
                queens[i] = bag[j];
                bag[j] = bag[lastIndex--];
            }
            return queens;
        }
    }
}