package ai.metaheuristic.genetic.selector;

import ai.metaheuristic.genetic.Individual;

/**
 * @author Yoann Eichelberger - M1 IWOCS
 */
public class StochasticDuelOperator<I extends Individual> implements SelectionOperator<I> {

    private I[] population;
    private float proba;

    /**
     * Construit une nouvelle instance de duel stochastique.
     * 
     * @param proba probabilité de tirer le meilleur individu.
     */
    public StochasticDuelOperator(float proba) {
        this.proba = proba;
    }

    /**
     * Stocke en mémoire la population.
     */
    @Override
    public void compute(I[] population) {
        this.population = population;
    }

    /**
     * Prends un couple d'individus dans la population, tire un nombre aléatoire
     * entre 0 et 1 et vérifie si ce dernier est inférieur à la probalité de tirer
     * le meilleur individu. Si c'est vérifié, renvoie le meilleur sinon le moins
     * bon.
     * 
     * @return un individu aléatoirement choisi dans un couple.
     */
    @Override
    public I random() {
        int n = this.population.length;
        int a = (int) (Math.floor(Math.random() * n));
        int b = (int) (a + 1 + Math.floor(Math.random() * n)) % n;
        I ia = this.population[a];
        I ib = this.population[b];
        double fa = ia.fitness();
        double fb = ib.fitness();
        I best = null, worst = null;
        if (fa > fb) {
            best = ia;
            worst = ib;
        } else if (fa < fb) {
            best = ib;
            worst = ia;
        } else if (fa == fb) {
            if (Math.random() < 0.5) {
                best = ia;
                worst = ib;
            } else {
                best = ib;
                worst = ia;
            }
        }
        return Math.random() < this.proba ? best : worst;
    }
}