package ai.metaheuristic.genetic.selector;

import java.util.function.Function;

import ai.metaheuristic.genetic.Individual;

/**
 * @author Yoann Eichelberger - M1 IWOCS
 */
public class BiasedWheelOperator<I extends Individual> implements SelectionOperator<I> {

    private Function<Double, Double> fitnessTransformation;
    private I[] population;
    private double sum;
    private double[] fitness;

    /**
     * Construit une nouvelle instance de roue biaisée.
     * 
     * @param fitnessTransformation opération de transformation de chaque fitness.
     */
    public BiasedWheelOperator(Function<Double, Double> fitnessTransformation) {
        this.fitnessTransformation = fitnessTransformation;
    }

    /**
     * Paramètre la roue de la forture et stocke en mémoire la population.
     */
    @Override
    public void compute(I[] population) {
        this.population = population;
        this.sum = 0;
        if (this.fitness == null || this.fitness.length != population.length) {
            this.fitness = new double[population.length];
        }
        for (int i = 0; i < this.population.length; i++) {
            double fit = this.fitnessTransformation.apply(population[i].fitness());
            this.fitness[i] = this.sum;
            this.sum += fit;
        }
    }

    /**
     * Va tirer un individu en fonction de sa fitness. Plus cette dernière est
     * élevée, plus l'individu aura de chances d'être tiré.
     * 
     * @return un individu tiré aléatoirement sur la roue de la forture.
     */
    @Override
    public I random() {
        return this.population[indexInInterval(this.fitness, 0, this.fitness.length - 1, Math.random() * this.sum)];
    }

    /**
     * Récupère un indice par dichotomie sur un interval donné.
     * 
     * @param a tableau trié croissant.
     * @param b début inclus du segment ciblé.
     * @param e fin inclus du segment ciblé.
     * @param v valeur à rechercher.
     * @return l'indice de la valeur la plus proche strictement inférieure à la
     *         valeur recherchée.
     */
    private static int indexInInterval(double[] a, int b, int e, double v) {
        if (e - b <= 1) {
            return v >= a[e] ? e : b;
        }
        int m = b + (e - b) / 2;
        return v < a[m] ? indexInInterval(a, b, m, v) : indexInInterval(a, m, e, v);
    }
}
