package ai.metaheuristic.genetic;

import java.util.Arrays;
import java.util.function.Function;

import ai.metaheuristic.genetic.selector.SelectionOperator;

/**
 * @author Yoann Eichelberger - M1 IWOCS
 */
public abstract class GeneticAlgorithm<I extends Individual> {

    private int length;
    private Function<Integer, I[]> arrayBuilder;
    private SelectionOperator<I> selectionOperator;
    private int bests;
    private int worsts;
    private int news;
    private int children;
    private float prcMutation;

    /**
     * Construit une instance d'algorithme génétique.
     */
    protected GeneticAlgorithm(int length, Function<Integer, I[]> arrayBuilder, SelectionOperator<I> selectionOperator,
            float prcBest, float prcWorst, float prcNew, float prcMutation) {
        this.length = length;
        this.arrayBuilder = arrayBuilder;
        this.selectionOperator = selectionOperator;
        this.bests = (int) Math.floor(length * prcBest);
        this.worsts = (int) Math.floor(length * prcWorst);
        this.news = (int) Math.floor(length * prcNew);
        this.children = length - bests - worsts - news;
        this.prcMutation = prcMutation;
    }

    /**
     * @return un nouvel individu.
     */
    public abstract I generate();

    /**
     * @param individual sur qui appliquer une mutation.
     * @return un individu muté.
     */
    public abstract I mutate(I individual);

    /**
     * Applique un croissement de gène sur deux individus.
     * 
     * @param ind1 parent 1.
     * @param ind2 parent 2.
     * @return un nouvel individu partageant les gènes de ses parents.
     */
    public abstract I crossover(I ind1, I ind2);

    /**
     * Lance l'algorithme génétique.
     * 
     * @return I un individu répondant au problème.
     */
    public I geneticAlgorithm() {
        // 1. Génération d’une population initiale d’individus.
        I[] population = this.arrayBuilder.apply(this.length);
        int n = this.length;
        for (int i = 0; i < n; i++) {
            population[i] = this.generate();
        }
        while (true) {
            // 2. Calcul du score de chaque individu.
            Arrays.parallelSort(population);
            // 3. Si l’objectif est atteint, sortie.
            if (population[0].fitness() >= 1) {
                return population[0];
            }
            I[] nextPopulation = this.arrayBuilder.apply(this.length);
            int i = 0;
            // 4. Sélection des reproducteurs en fonction des scores.
            this.selectionOperator.compute(population);
            for (int k = 0; k < this.children; k++, i++) {
                I ind1 = this.selectionOperator.random();
                I ind2 = this.selectionOperator.random();
                // 5. Construction des descendants par application de différents opérateurs
                // génétiques (étape de reproduction).
                I child = this.crossover(ind1, ind2);
                if (this.prcMutation > 0) {
                    double rdm = Math.random() * (child.length() * this.prcMutation);
                    for (int j = 0; j < rdm; j++) {
                        child = this.mutate(child);
                    }
                }
                nextPopulation[i] = child;
            }
            // 6. Mélange avec les descendants.
            // % des plus aptes de la population parente.
            for (int k = 0; k < this.bests; k++, i++) {
                nextPopulation[i] = population[k];
            }
            // % des moins aptes de la population parente.
            for (int k = 0; k < this.worsts; k++, i++) {
                nextPopulation[i] = population[n - k - 1];
            }
            // % de nouveaux individus.
            for (int k = 0; k < this.news; k++, i++) {
                nextPopulation[i] = this.generate();
            }
            population = nextPopulation;
            Arrays.parallelSort(population);
            // Goto 3.
        }
    }
}