const Ad = require('../models/Ad');
const User = require('../models/User');
const fs = require('fs');
const path = require('path');
const {extractUserFromToken} = require("./UserService");

const secretKey = 'secretKey';

// Service for adding an ad
exports.addAdWithPhotos = async function (body, photos, token) {
    try {
        const { title, propertyType, publicationStatus, propertyStatus, description, price, availabilityDate } = body;
        const user = await extractUserFromToken(token, secretKey);

        const photoPaths = photos.map(file => {
            // Process and store the file, return its path
            const photoFileName = `${Date.now()}_${file.originalname}`;
            const relativePath = path.join('uploads', photoFileName);
            fs.writeFileSync(relativePath, file.buffer);
            return relativePath;
        });

        const newAnnouncement = new Ad({
            title,
            propertyType,
            publicationStatus,
            propertyStatus,
            description,
            price,
            availabilityDate,
            photos: photoPaths,
            userName: user.username
        });

        await newAnnouncement.save();

        return newAnnouncement;
    } catch (error) {
        console.error('Error adding ad with photos:', error);
        throw new Error('Failed to add ad with photos. Check server logs for details.');
    }
};


// Service for adding photos to an ad
exports.addPhotosToAd = async function (announcementId, photos,token) {
    try {
        console.log(announcementId, ' ', photos);
        const user = await extractUserFromToken(token, secretKey);
        const photoPaths = photos.map(file => {
            const photoFileName = `${Date.now()}_${file.originalname}`;
            const relativePath = path.relative(process.cwd(), path.join(process.cwd(), 'uploads', photoFileName));
            fs.writeFileSync(relativePath, file.buffer);
            return relativePath;
        });

        const updatedAnnouncement = await Ad.findByIdAndUpdate(
            announcementId,
            { $push: { photos: { $each: photoPaths } } },
            { new: true }
        );

        if (!updatedAnnouncement) {
            return { status: 404, message: 'Ad not found.' };
        }

        return { status: 200, message: 'Photos added successfully.', updatedAd: updatedAnnouncement };
    } catch (error) {
        throw error;
    }
};

// Service for updating an ad with photos
exports.updateAd = async function (announcementId, reqBody, files, token) {
    try {
        const user = await extractUserFromToken(token, secretKey);
        const { title, propertyType, publicationStatus, description, price, availabilityDate } = reqBody;

        const photoPaths = files.map(file => {
            const photoFileName = `${Date.now()}_${file.originalname}`;
            const relativePath = path.join('uploads', photoFileName);
            fs.writeFileSync(relativePath, file.buffer);
            return relativePath;
        });

        const updatedAnnouncement = await Ad.findByIdAndUpdate(
            announcementId,
            {
                title,
                propertyType,
                publicationStatus,
                description,
                price,
                availabilityDate,
                $push: { photos: { $each: photoPaths } }
            },
            { new: true }
        );

        if (!updatedAnnouncement) {
            return { status: 404, message: 'Annonce non trouvée.' };
        }

        if (user.username !== updatedAnnouncement.userName) {
            return { status: 401, message: 'Utilisateur non autorisé.' };
        }

        return { status: 200, message: 'Annonce modifiée avec succès.', updatedAd: updatedAnnouncement };
    } catch (error) {
        console.error('Error updating ad with photos:', error);
        throw new Error('Failed to update ad with photos. Check server logs for details.');
    }
};

exports.deleteAd = function (announcementId, token) {
  return new Promise(async (resolve, reject) => {
    try {
        const user = await extractUserFromToken(token, secretKey);
        const Announcement = await Ad.findById(announcementId);
        if (user.username !== Announcement.userName) {
            return resolve({ status: 403, message: 'User non autorisé' });
        }

      const deletedAnnouncement = await Ad.findByIdAndRemove(announcementId);

      if (!deletedAnnouncement) {
        return resolve({ status: 404, message: 'annonce non trouvée' });
      }

      /*
      deletedAnnouncement.photos.forEach((photo) => {
        const filePath = path.join('public', 'uploads', photo);
        if (fs.existsSync(filePath)) {
          fs.unlinkSync(filePath);
        }
      });
      */
      resolve({ status: 200, message: 'Annonce supprimée avec succès' });
    } catch (error) {
      reject(error);
    }
  });
};

exports.getAds = function (token) {
  return new Promise(async (resolve, reject) => {
    try {
        const decoded = await extractUserFromToken(token, secretKey);
        let announcements;
        const user = await User.findOne({ username: decoded.username });
        if (user && user.isAgent) {
            announcements = await Ad.find({ userName: user.username });
        } else {
            announcements = await Ad.find({ publicationStatus: 'Publiée' });
        }
        resolve(announcements);
    } catch (error) {
      reject(error);
    }
  });
};

exports.getAdById = function (announcementId, token) {
  return new Promise(async (resolve, reject) => {
    try {
        const announcementDetails = await Ad.findById(announcementId);
        const decoded = await extractUserFromToken(token, secretKey);
        const user = await User.findOne({ username: decoded.username });


        if (
          (user && user.username !== announcementDetails.userName || !user) &&
          announcementDetails.publicationStatus !== 'Publiée'
          ) {
            return resolve({ status: 404, message: 'Cette annonce est introuvable ou non disponible.' });
          }
          if (user && user.username !== announcementDetails.userName && user.isAgent) {
            return resolve({ status: 403, message: 'User non autorisée.' });
          }

          if( !announcementDetails ) {
            return resolve({ status: 404, message: 'Cette annonce est introuvable ou non disponible.' });
          }
          resolve(announcementDetails);
    } catch (error) {
      reject(error);
    }
  });
};

// Service pour poser une question sur une annonce
exports.askQuestion = async function(announcementId, question, token) {
  return new Promise(async (resolve, reject) => {
      try {
      const announcement = await Ad.findById(announcementId);
      const decoded = await extractUserFromToken(token, secretKey);
      const user = await User.findOne({ username: decoded.username });

      if (!announcement) {
        return resolve({ status: 404, message: 'Annonce non trouvée.' });
      }
      if (announcement.propertyStatus !== 'Disponible') {
        return resolve({ status: 404, message: 'Vous ne pouvez poser une question que sur les annonces non disponibles.' });
      }
      announcement.questions.push({
        user: user.username,
        question
      });
      await announcement.save();
          console.log(announcement, 'hhhhhhhhh')
      resolve({ status: 200, message: 'Question posée avec succès.' });
    } catch (error) {
      throw error;
    }
  });
};

// Service pour répondre à une question sur une annonce
exports.answerQuestion = async function(announcementId, questionId, answer, token) {
  return new Promise(async (resolve, reject) => {
      try {
      const announcement = await Ad.findById(announcementId);
      const decoded = await extractUserFromToken(token, secretKey);
      const user = await User.findOne({ username: decoded.username });

      if (!announcement) {
        return resolve({ status: 404, message: 'Annonce non trouvée.' });
      }

      if (user.username !== announcement.userName) {
          return resolve({ status: 403, message: 'User non autorisée.' });
      }

      const question = announcement.questions.id(questionId);
      if (!question) {
        return resolve({ status: 404, message: 'Question non trouvée.' });
      }
      question.answers.push({ user: user.username, answer });
      await announcement.save();
      resolve({ status: 200, message: 'Réponse ajoutée avec succès.' });
    } catch (error) {
      throw error;
    }
  });
}
