const { makeExecutableSchema } = require("@graphql-tools/schema");
const UserModel = require("../models/User");
const axios = require('axios');
const {
    addAd,
    updateAd,
    getAdById,
    getAds,
    deleteAd,
    askQuestion,
    answerQuestion,
} = require("../service/AdService");

const typeDefs = `
  type Ad {
    _id: ID!
    title: String!
    propertyType: String!
    publicationStatus: String!
    propertyStatus: String!
    description: String!
    price: Float!
    availabilityDate: String!
    photos: [String]
    userName: String!
    questions: [Question]
  }

  type Question {
    user: String!
    question: String!
    answers: [Answer]
    date: String!
  }

  type Answer {
    user: String!
    answer: String!
  }

  type AuthPayload {
    token: String!
  }

  type Query {
    getAds(token: String): [Ad]
    getAdById(announcementId: ID!, token: String): Ad
    getUser(id: ID, username: String): User
  }

  type Mutation {
    addAd(input: AdInput, token: String): Ad
    updateAd(announcementId: ID!, input: AdInput, token: String): Ad
    deleteAd(announcementId: ID!, token: String): String
    askQuestion(announcementId: ID!, question: String!, token: String): String
    answerQuestion(
      announcementId: ID!
      questionId: ID!
      answer: String!
      token: String
    ): String
    createUser(input: UserInput): User
    loginUser(username: String!, password: String!): AuthPayload
    logoutUser: String
  }

  input AdInput {
    title: String!
    propertyType: String!
    publicationStatus: String!
    propertyStatus: String!
    description: String!
    price: Float!
    availabilityDate: String!
    photos: [String]
  }

  type User {
    _id: ID!
    username: String!
    password: String
    isAgent: Boolean
  }

  input UserInput {
    username: String!
    password: String
    isAgent: Boolean
  }
`;

const resolvers = {
    Query: {
        getAds: (_, { token }) => getAds(token),
        getAdById: (_, { announcementId, token }) => getAdById(announcementId, token),
        getUser: async (parent, { id, username }) => {
            if (id) {
                return await UserModel.findById(id);
            } else if (username) {
                return await UserModel.findOne({ username });
            } else {
                return null;
            }
        },
    },
    Mutation: {
        addAd: (_, { input, token }) => addAd(input, token),
        updateAd: async (_, { announcementId, input, token }) => {
            try {
                const result = await updateAd(announcementId, input, token);

                if (result.status === 200) {
                    return result.updatedAd;
                } else {
                    throw new Error(result.message);
                }
            } catch (error) {
                throw new Error('An error occurred while updating the ad.');
            }
        },
        deleteAd: async (_, { announcementId, token }) => {
            try {
                const result = await deleteAd(announcementId, token);

                if (result.status === 200) {
                    return {
                        success: true,
                        message: result.message,
                    };
                } else {
                    return {
                        success: false,
                        message: result.message,
                    };
                }
            } catch (error) {
                return {
                    success: false,
                    message: 'An error occurred while deleting the ad.',
                    error: error.message,
                };
            }
        },
        askQuestion: async (_, { announcementId, question, token }) => {
            try {
                const result = await askQuestion(announcementId, question, token);

                if (result.status === 200) {
                    return {
                        success: true,
                        message: result.message,
                    };
                } else {
                    return {
                        success: false,
                        message: result.message,
                    };
                }
            } catch (error) {
                return {
                    success: false,
                    message: 'An error occurred while asking the question.',
                    error: error.message,
                };
            }
        },
        answerQuestion: (_, { announcementId, questionId, answer, token }) =>
            answerQuestion(announcementId, questionId, answer, token),
        createUser: async (_, { input }) => {
            try {
                const response = await axios.post('http://localhost:8080/user', input);
                if (response.data.exports) {
                    throw new Error(response.data.exports.error);
                }
                return response.data.user;
            } catch (error) {
                console.error('Erreur lors de la création de l\'utilisateur:', error);
                throw new Error('Erreur lors de la création de l\'utilisateur. Vérifiez les détails dans les journaux.');
            }
        },
        loginUser: async (_, { username, password }) => {
            try {
                const response = await axios.post('http://localhost:8080/user/login', { username, password });

                if (response.status === 200) {
                    return { token: response.data.token };
                } else {
                    throw new Error('Invalid username or password.');
                }
            } catch (error) {
                console.error('Erreur lors de la tentative de connexion:', error);
                throw new Error('Erreur lors de la tentative de connexion. Vérifiez les détails dans les journaux.');
            }
        },
        logoutUser: async () => {
            try {
                const response = await axios.post('http://localhost:8080/user/logout');

                if (response.status === 200) {
                    return 'Logout successful.';
                } else {
                    throw new Error('Logout failed.');
                }
            } catch (error) {
                console.error('Erreur lors de la déconnexion de l\'utilisateur:', error);
                throw new Error('Erreur lors de la déconnexion de l\'utilisateur. Vérifiez les détails dans les journaux.');
            }
        },
    },
};

exports.schema = makeExecutableSchema({ typeDefs, resolvers });
