const express = require('express');
const swaggerUi = require('swagger-ui-express');
const {makeExecutableSchema} = require('@graphql-tools/schema');
const {createHandler} = require('graphql-http/lib/use/express');
const expressPlayground = require('graphql-playground-middleware-express').default;
const YAML = require('yamljs');
const swaggerJsdoc = require('swagger-jsdoc');
const swaggerParser = require('swagger-parser');
const connectToDatabase = require('./data-access/connection');
const apiRouter = require('./controllers/Ad');
const authRouter = require('./controllers/User');
const path = require('path');

const app = express();
const serverPort = 8080;

// Middleware for parsing JSON
app.use(express.json());

const options = {
    definition: {
        openapi: '3.0.3',
        info: {
            title: 'Api Ads',
            version: '1.0.0',
            description: 'Ads API',
        },
    },
    apis: ['./controllers/*.js'],
};

const openApiPath = path.join(__dirname, 'api', 'openapi.yaml');
const openApiDocument = YAML.load(openApiPath);

swaggerParser.validate(openApiDocument, (err, api) => {
    if (err) {
        console.error('Error validating OpenAPI document:', err);
        process.exit(1);
    } else {
        console.log('OpenAPI document is valid');
    }
});

const swaggerSpec = swaggerJsdoc(options);

const {schema} = require("./schemas/User&AdSchema");

app.use('/docs', swaggerUi.serve, swaggerUi.setup(openApiDocument, swaggerSpec));
app.get('/graphql-playground', expressPlayground({endpoint: '/graphql'}));
app.post('/graphql', createHandler({schema}));

app.use('/announcements', apiRouter);
app.use('/user', authRouter);
app.use(express.json());

connectToDatabase()
    .then(() => {
        app.listen(serverPort, () => {
            console.log(`Your server is listening on port ${serverPort} (http://localhost:${serverPort})`);
            console.log('Swagger-ui is available on http://localhost:%d/docs', serverPort);
            console.log('GraphQL Playground is available on http://localhost:%d/graphql-playground', serverPort);
        });
    })
    .catch(error => {
        console.error('Error connecting to the database:', error);
    });
