import 'package:flutter/material.dart';
import 'package:googleapis_auth/auth_io.dart' as auth;
import 'package:googleapis/sheets/v4.dart' as sheets;
import 'dart:io' show File;

import 'config.dart';

class StudentEditPage extends StatefulWidget {
  final String googleSheetsLink;

  const StudentEditPage({Key? key, required this.googleSheetsLink}) : super(key: key);

  @override
  _StudentEditPageState createState() => _StudentEditPageState();
}

class _StudentEditPageState extends State<StudentEditPage> {
  final TextEditingController _firstNameController = TextEditingController();
  final TextEditingController _lastNameController = TextEditingController();
  final TextEditingController _studentNumberController = TextEditingController();
  String? _spreadsheetId;

  @override
  void initState() {
    super.initState();
    _extractSpreadsheetId();
  }

  void _extractSpreadsheetId() {
    final regExp = RegExp(r'/spreadsheets/d/([a-zA-Z0-9-_]+)');
    final matches = regExp.firstMatch(widget.googleSheetsLink);
    if (matches != null && matches.groupCount >= 1) {
      _spreadsheetId = matches.group(1);
    } else {
      print("L'URL de la feuille Google Sheets est invalide");
    }
  }

  Future<void> _addStudentToSheet() async {
    if (_firstNameController.text.isEmpty || _lastNameController.text.isEmpty || _studentNumberController.text.isEmpty) {
      ScaffoldMessenger.of(context).showSnackBar(const SnackBar(
        content: Text("Veuillez remplir tous les champs avant de soumettre."),
        backgroundColor: Colors.redAccent,
      ));
      return;
    }
    if (_spreadsheetId == null) {
      print("Id de la feuille Google Sheets non trouve");
      return;
    }

    final accountCredentials = auth.ServiceAccountCredentials.fromJson(
        File(AppConfig.cleCompteService).readAsStringSync());

    final scopes = [sheets.SheetsApi.spreadsheetsScope];

    try {
      final authClient = await auth.clientViaServiceAccount(accountCredentials, scopes);
      var api = sheets.SheetsApi(authClient);
      var range = '${AppConfig.sheetName}!A:A';
      var valueRange = sheets.ValueRange.fromJson({
        "values": [
          [
            _firstNameController.text,
            _lastNameController.text,
            _studentNumberController.text,
            "", //vide pour le moment "Code NFC"
            "oui" // Présence
          ]
        ]
      });

      await api.spreadsheets.values.append(valueRange, _spreadsheetId!, range,
          valueInputOption: 'USER_ENTERED');

      ScaffoldMessenger.of(context).showSnackBar(const SnackBar(
        content: Text("L'étudiant a été ajouté avec succès à la feuille."),
      ));
      authClient.close();
    } catch (e) {
      print(e);
      ScaffoldMessenger.of(context).showSnackBar(const SnackBar(
        content: Text("Erreur lors de l'ajout de l'étudiant à la feuille."),
        backgroundColor: Colors.red,
      ));
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text("Ajouter un étudiant"),
        backgroundColor: Colors.deepPurple,
      ),
      body: SingleChildScrollView(
        child: Padding(
          padding: const EdgeInsets.all(16.0),
          child: Column(
            children: [
              const Text(
                "Informations de l'étudiant",
                style: TextStyle(fontSize: 24, fontWeight: FontWeight.bold, color: Colors.deepPurple),
              ),
              const SizedBox(height: 20),
              _buildTextField(_firstNameController, "Prénom de l'étudiant"),
              const SizedBox(height: 10),
              _buildTextField(_lastNameController, "Nom de l'étudiant"),
              const SizedBox(height: 10),
              _buildTextField(_studentNumberController, "Numéro de l'étudiant"),
              const SizedBox(height: 30),
              ElevatedButton(
                onPressed: _addStudentToSheet,
                style: ElevatedButton.styleFrom(
                  primary: Colors.deepPurple,
                  onPrimary: Colors.white,
                  padding: EdgeInsets.symmetric(horizontal: 50, vertical: 20),
                ),
                child: Text("Ajouter à la Google Sheet"),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildTextField(TextEditingController controller, String label) {
    return TextField(
      controller: controller,
      decoration: InputDecoration(
        labelText: label,
        border: OutlineInputBorder(
          borderRadius: BorderRadius.circular(10.0),
        ),
        fillColor: Colors.deepPurple.shade50,
        filled: true,
      ),
    );
  }
}
