import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:nfc/student_edit_page.dart';
import 'package:open_file/open_file.dart';
import 'package:pdf/widgets.dart' as pw;
import 'config.dart';
import 'dart:io';
import 'nfc_service.dart';

class DataPage extends StatefulWidget {
  final String googleSheetsLink;

  const DataPage({Key? key, required this.googleSheetsLink}) : super(key: key);

  @override
  _DataPageState createState() => _DataPageState();
}

class _DataPageState extends State<DataPage> {
  List<List<dynamic>> _sheetData = [];
  late bool _isLoading;

  @override
  void initState() {
    super.initState();
    _isLoading = true;
    _loadSheetData();
  }

  Future<void> _loadSheetData() async {
    RegExp regex = RegExp(
        r'https:\/\/docs\.google\.com\/spreadsheets\/d\/([a-zA-Z0-9_-]+)\/(edit|pubhtml).*');
    Match? match = regex.firstMatch(widget.googleSheetsLink);

    if (match != null) {
      String sheetId = match.group(1)!;
      String apiUrl =
          'https://sheets.googleapis.com/v4/spreadsheets/$sheetId/values/${AppConfig.sheetName}?key=${AppConfig.apiKey}';

      try {
        final response = await http.get(Uri.parse(apiUrl));

        if (response.statusCode == 200) {
          final Map<String, dynamic> data = json.decode(response.body);
          List<List<dynamic>> values = [];

          if (data['values'] == null) {
            setState(() {
              _isLoading = false;
            });
            return;
          }

          for (final entry in data['values']) {
            values.add(List.from(entry));
          }

          setState(() {
            _sheetData = values;
          });
        } else {
          print(
              'Erreur lors de la récupération des données du Google Sheets. Statut : ${response.statusCode}');
        }
      } catch (e) {
        print(
            'Erreur lors de la récupération des données du Google Sheets : $e');
      }
    }
  }

  Future<void> generatePdf(List<List<dynamic>> sheetData) async {
    final pdf = pw.Document();

    final List<pw.TableRow> rows = [];

    for (final row in sheetData) {
      final List<pw.Widget> rowWidgets = row.map((cell) => pw.Text('$cell')).toList();
      rows.add(pw.TableRow(children: rowWidgets));
    }

    pdf.addPage(pw.Page(
      build: (pw.Context context) {
        return pw.Table(
          columnWidths: {
            for (int i = 0; i < sheetData[0].length; i++)
              i: const pw.FixedColumnWidth(100.0),
          },
          children: rows,
        );
      },
    ));

    final file = File('google_sheets_data.pdf');
    await file.writeAsBytes(await pdf.save());

    OpenFile.open(file.path);
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text('Page de données'),
        backgroundColor: Colors.deepPurple,
        centerTitle: true,
      ),
      body: SingleChildScrollView(
        child: Padding(
          padding: const EdgeInsets.all(16.0),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.center,
            children: <Widget>[
              if (_sheetData.isNotEmpty)
                Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: <Widget>[
                    const Text(
                      'Contenu du Google Sheets :',
                      style:
                      TextStyle(fontSize: 18.0, fontWeight: FontWeight.bold, color: Colors.deepPurple),
                    ),
                    const SizedBox(height: 10.0),
                    Table(
                      border: TableBorder.all(),
                      columnWidths: {
                        for (int i = 0; i < _sheetData[0].length; i++)
                          i: const FixedColumnWidth(100.0),
                      },
                      children: _sheetData.map((row) {
                        if (row.length < _sheetData[0].length) {
                          final filledRow = List.from(row)
                            ..addAll(List.generate(
                                _sheetData[0].length - row.length, (_) => ''));

                          return TableRow(
                            children: filledRow.map((cell) => Padding(
                              padding: const EdgeInsets.all(8.0),
                              child: Text(
                                '$cell',
                                style: const TextStyle(fontWeight: FontWeight.bold),
                              ),
                            )).toList(),
                          );
                        } else {
                          return TableRow(
                            children: row.map((cell) => Padding(
                              padding: const EdgeInsets.all(8.0),
                              child: Text(
                                '$cell',
                                style: const TextStyle(fontWeight: FontWeight.bold),
                              ),
                            )).toList(),
                          );
                        }
                      }).toList(),
                    ),
                    const SizedBox(height: 20.0),
                    Row(
                      children: <Widget>[
                        ElevatedButton(
                          onPressed: () async {
                            await generatePdf(_sheetData);
                          },
                          style: ElevatedButton.styleFrom(
                            primary: Colors.deepPurple,
                            onPrimary: Colors.white,
                          ),
                          child: const Text('Générer PDF'),
                        ),
                        const SizedBox(width: 20.0),
                        ElevatedButton(
                          onPressed: () async {
                            String? nfcData = await NFCService.readNFC();
                            if (nfcData != null) {
                              // Traitez les données NFC ici
                              print('Données NFC lues: $nfcData');
                            } else {
                              // Gérer l'erreur de lecture NFC
                              print('Erreur lors de la lecture NFC');
                            }
                          },
                          style: ElevatedButton.styleFrom(
                            primary: Colors.deepPurple,
                            onPrimary: Colors.white,
                          ),
                          child: const Text('Lire la leocarte'),
                        ),
                        const SizedBox(width: 20.0),
                        ElevatedButton(
                          onPressed: () {
                            Navigator.push(
                              context,
                              MaterialPageRoute(
                                builder: (context) => StudentEditPage(googleSheetsLink: widget.googleSheetsLink),
                              ),
                            );
                          },
                          style: ElevatedButton.styleFrom(
                            primary: Colors.deepPurple,
                            onPrimary: Colors.white,
                          ),
                          child: const Text('Ajouter un étudiant'),
                        ),
                      ],
                    ),
                  ],
                )
              else if (_isLoading)
                const Text(
                  'Chargement des données..',
                  style: TextStyle(fontSize: 16.0),
                )
              else if (_sheetData.isEmpty)
                  const Text(
                    'Aucune donnée disponible.',
                    style: TextStyle(fontSize: 16.0),
                  ),
            ],
          ),
        ),
      ),
    );
  }
}