'use strict'

//Fabric smart contract classes
const{Contract, Context} = require('fabric-contract-api')

// Car market specific classes
const Car = require('./car.js');
const CarList = require('./carlist.js');

/**
 * A custom context provides easy access to list of all commercial papers
 */
class CarContext extends Context {

    constructor() {
        super();
        // All cars are held in a list of cars
        this.carList = new CarList(this);
    }
}

/**
 * Define a car smart contract by extending Fabric Contract class
 */
class CarContract extends Contract{
    constructor(){
        super('org.carmarket.car')
    }

    createContext() {
        return new CarContext();
    }
    
    /**
     * Instantiate to perform any setup of the ledger that might be required.
     * @param {Context} ctx the transaction context
     */
    async instantiate(ctx) {
        // No implementation required with this example
        // It could be where data migration is performed, if necessary
        console.log('Instantiate the contract');
    }

    async sale(ctx, seller, carNumber, carBrand, carModel, price){
        // Create an instance of the car
        let car = Car.createInstance(seller, carNumber, carBrand, carModel, price)
        
        // Set the car into the forsale state
        car.setForSale();

        car.setOwner(seller);
        
        // Add the car to the list of cars into the ledger state
        await ctx.carList.addCar(car);
        
        return car;
    }

    async buy(ctx, seller, carNumber, currentOwner, newOwner, price){
        //Retrieve the car
        let carKey = Car.makeKey([seller, carNumber]);
        let car = await ctx.carList.getCar(carKey);

        //Validate owner
        if(car.getOwner() !== currentOwner){
            throw new Error('Car ' + seller + carNumber + ' is not owned by ' + currentOwner);
        }

        //Check if car isnt already sold
        if(car.isSold()){
            throw new Error('Car ' + seller + carNumber + ' is already sold. Current state = ' + paper.getCurrentState());
        }

        //Check if the price correspond
        if(price !== car.getPrice()){
            throw new Error('Wrong price entered, the price of this car is : ' + car.getPrice());
        }

        //Set the car into sold state
        if(car.isForSale()){
            car.setSold();
            car.setOwner(newOwner);
        }
        await ctx.carList.updateCar(car);
        return car;
    }

    async info(ctx, seller, carNumber){
        let carKey = Car.makeKey([seller, carNumber]);
        let car = await ctx.carList.getCar(carKey);

        return car;
    }
}
module.exports = CarContract;