'use client';

import Link from 'next/link';
import { useState } from 'react';
import { signIn } from 'next-auth/react';
import { useRouter } from 'next/navigation';

export default function Login() {
    const [email, setEmail] = useState('');
    const [password, setPassword] = useState('');
    const [error, setError] = useState<string | null>(null);
    const router = useRouter();

    // --- Gestionnaire de soumission du formulaire ---
    const handleSubmit = async (event: React.FormEvent<HTMLFormElement>) => {
        event.preventDefault(); // Empêche le rechargement de la page
        setError(null); // Réinitialise les erreurs

        try {
            // --- Appel à NextAuth ---
            const result = await signIn('credentials', {
                redirect: false,
                email: email,
                password: password,
            });

            if (result?.error) {
                // mauvais mot de passe ou mauvais email
                setError("L'adresse e-mail ou le mot de passe est incorrect.");
                console.error('Failed to sign in:', result.error);
            } else if (result?.ok) {
                // connexion réussie
                router.push('/');
            }
        } catch (error) {
            setError("Une erreur est survenue. Veuillez réessayer.");
            console.error('Sign in error:', error);
        }
    };

    return (
        <div className="font-sans flex flex-col items-center justify-center min-h-screen p-4">
            <div className="w-full max-w-md">
                <header className="mb-8 text-center">
                    <nav className="mb-6">
                        <ul className="flex justify-center space-x-4">
                            <li>
                                <Link href="/" className="border rounded px-4 py-2 block hover:bg-gray-200 hover:text-gray-800">
                                    Accueil
                                </Link>
                            </li>
                            <li>
                                <Link href="/register" className="border rounded px-4 py-2 block hover:bg-gray-200 hover:text-gray-800">
                                    S'inscrire
                                </Link>
                            </li>
                        </ul>
                    </nav>
                    <h1 className="text-4xl font-bold mb-2">Connexion</h1>
                    <p className="text-lg text-gray-600">Accédez à votre compte.</p>
                </header>

                {/* --- Le Formulaire de Connexion --- */}
                <form onSubmit={handleSubmit} className="space-y-6">
                    <div>
                        <label htmlFor="email" className="block text-sm font-medium text-gray-700">
                            Adresse e-mail
                        </label>
                        <input
                            id="email"
                            name="email"
                            type="email"
                            autoComplete="email"
                            required
                            value={email}
                            onChange={(e) => setEmail(e.target.value)}
                            className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm placeholder-gray-400 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                        />
                    </div>

                    <div>
                        <label htmlFor="password" className="block text-sm font-medium text-gray-700">
                            Mot de passe
                        </label>
                        <input
                            id="password"
                            name="password"
                            type="password"
                            autoComplete="current-password"
                            required
                            value={password}
                            onChange={(e) => setPassword(e.target.value)}
                            className="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm placeholder-gray-400 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                        />
                    </div>

                    {/* --- Affichage de l'erreur --- */}
                    {error && (
                        <p className="text-sm text-red-600 bg-red-100 p-3 rounded-md">{error}</p>
                    )}

                    <div>
                        <button
                            type="submit"
                            className="w-full flex justify-center py-2 px-4 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                        >
                            Se connecter
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
}