import { Annonce } from "@prisma/client";
import prisma from "@/lib/global/globalPrisma";
import { AnnonceWithImagesProps } from "../types/AnnonceProps";

export const createAnnonce = async (data: {
    agentId: number;
    title: string;
    address: string;
    ville: string;
    codePostal: string;
    pays: string;
    description: string;
    prix: number;
    imagePrincipale?: string;
}): Promise<Annonce> => {
    const annonce = await prisma.annonce.create({
        data: {
            agentId: data.agentId,
            titre: data.title,
            address: data.address,
            ville: data.ville,
            codePostal: data.codePostal,
            pays: data.pays,
            description: data.description,
            prix: data.prix,
            imagePrincipale: data.imagePrincipale ?? null,
        },
    });
    return annonce;
};

export const getAnnonceById = async (id: number): Promise<Annonce | null> => {
    return await prisma.annonce.findUnique({
        where: { id },
    });
};

export const getAnnonceByIdInclude = async (id: number): Promise<AnnonceWithImagesProps | null> => {
    return prisma.annonce.findUnique({
        where: { id },
        include: {
            autresImages: { },
        },
    });
};

export const updateAnnonce = async (
    id: number,
    data: {
        title?: string;
        address?: string;
        ville?: string;
        codePostal?: string;
        pays?: string;
        description?: string;
        prix?: number;
    }
): Promise<Annonce> => {
    return await prisma.annonce.update({
        where: { id },
        data: {
            titre: data.title,
            address: data.address,
            ville: data.ville,
            codePostal: data.codePostal,
            pays: data.pays,
            description: data.description,
            prix: data.prix,
        },
    });
};

export const deleteAnnonce = async (id: number): Promise<Annonce> => {
    return await prisma.annonce.delete({
        where: { id },
    });
};

export const getAllAnnonces = async (): Promise<Annonce[]> => {
    return await prisma.annonce.findMany();
};

type EditableAnnonceFields = Pick<
    Annonce,
    "titre" | "address" | "ville" | "codePostal" | "pays" | "description" | "prix" | "statut" | "imagePrincipale"
>;

export const editAnnonce = (id: number, data: Partial<EditableAnnonceFields>): Promise<Annonce> => {
    return prisma.annonce.update({
        where: { id },
        data,
    });
};

export const editAnnonceImage = (id: number, base64Image: string): Promise<Annonce> => {
    return prisma.annonce.update({
        where: { id },
        data: { imagePrincipale: base64Image },
    });
};
