import { NextRequest, NextResponse } from "next/server";
import { createAnnonce, editAnnonce, getAllAnnonces } from "@/lib/services/annonceService";
import { getServerSession } from "next-auth";
import { authOptions } from "../auth/[...nextauth]/route";

export async function POST(req: NextRequest) {
    try {
        const session = await getServerSession(authOptions);
        if (!session) {
            return NextResponse.json({ message: "Non authentifié." }, { status: 401 });
        }
        
        const role = session.user?.role as string | undefined;

        if (!role || !["ADMIN", "AGENT"].includes(role)) {
            return NextResponse.json({ message: "Accès refusé." }, { status: 403 });
        }

        const formData = await req.formData();

        const agentId = session?.user?.id;
        const title = formData.get("title") as string | null;
        const address = formData.get("address") as string | null;
        const pays = formData.get("pays") as string | null;
        const ville = formData.get("ville") as string | null;
        const codePostal = formData.get("codePostal") as string | null;
        const description = formData.get("description") as string | null;
        const prix = formData.get("prix") as string | null;

        const files = formData.getAll("images") as File[];
        const file = files[0] ?? null;

        if (!agentId || !title || !prix || !file) {
            return NextResponse.json({ message: "Les champs requis sont manquants." }, { status: 400 });
        }

        const ALLOWED = new Set(["image/jpeg", "image/png", "image/webp"]);
        if (!ALLOWED.has(file.type)) {
            return NextResponse.json({ message: "Type d’image non autorisé." }, { status: 415 });
        }

        const arrayBuffer = await file.arrayBuffer();
        const base64Image = Buffer.from(arrayBuffer).toString("base64");

        const newAnnonce = await createAnnonce({
            agentId: Number(agentId),
            title: title,
            address: address ?? "",
            country: pays ?? "",
            city: ville ?? "",
            postalCode: codePostal ?? "",
            description: description ?? "",
            price: Number(prix),
            mainImg: base64Image,
        });

        return NextResponse.json(newAnnonce, { status: 201 });
    } catch (error) {
        console.error("Erreur API lors de la création de l'annonce:", error);
        return NextResponse.json({ message: "Erreur interne du serveur." }, { status: 500 });
    }
}
