import { CameraType, CameraView, useCameraPermissions, BarcodeScanningResult } from 'expo-camera';
import React, { useState } from 'react';
import { Button, StyleSheet, Text, View, Modal, TextInput, Image, ActivityIndicator, Alert } from 'react-native';
import ItemService from '../services/ItemService';
import { Item } from '../types/Item';
import { API_BASE_URL } from "../config";

export default function ScannerScreen() {
    const [facing, setFacing] = useState<CameraType>('back');
    const [permission, requestPermission] = useCameraPermissions();
    const [scanned, setScanned] = useState(false);
    const [scannedData, setScannedData] = useState<string | null>(null);

    const [item, setItem] = useState<Item | null>(null);
    const [isModalVisible, setIsModalVisible] = useState(false);
    const [isLoading, setIsLoading] = useState(false);
    const [newPrice, setNewPrice] = useState('');

    if (!permission) {
        return <View />;
    }

    if (!permission.granted) {
        return (
            <View style={styles.container}>
                <Text style={styles.permissionText}>Nous avons besoin de votre permission pour utiliser la caméra</Text>
                <Button onPress={requestPermission} title="Accorder la permission" />
            </View>
        );
    }

    const handleBarcodeScanned = async (scanningResult: BarcodeScanningResult) => {
        if (scanned) {
            return;
        }

        setScanned(true);
        setScannedData(scanningResult.data);
        setIsLoading(true);
        setNewPrice('');

        try {
            const service = new ItemService(scanningResult.data);
            const fetchedItem = await service.getItem();
            setItem(fetchedItem);
            setIsModalVisible(true);

        } catch (error) {
            console.error("Erreur lors de la récupération de l'article:", error);
            Alert.alert("Erreur", `Impossible de récupérer les détails pour le code-barres ${scanningResult.data}.`);
            setScanned(false);
            setScannedData(null);
        } finally {
            setIsLoading(false);
        }
    };

    const handleCloseModal = () => {
        setIsModalVisible(false);
        setItem(null);
        setScanned(false);
        setScannedData(null);
    };

    const handlePriceSubmit = async () => {
        if (!item || !newPrice) {
            Alert.alert("Erreur", "Veuillez entrer un prix valide.");
            return;
        }

        const priceValue = parseFloat(newPrice.replace(',', '.'));
        if (isNaN(priceValue)) {
            Alert.alert("Erreur", "Le prix doit être un nombre.");
            return;
        }

        if (priceValue <= 0) {
            Alert.alert("Erreur", "Le prix doit être supérieur à 0.");
            return;
        }

        try {
            const barcodeId = item.getBarcodeId();
            const url = `${API_BASE_URL}/items/barcode/${barcodeId}`;
            console.log(url);
            const response = await fetch(url, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify({
                    price: priceValue,
                }),
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.detail || "Erreur lors de la mise à jour du prix");
            }

            const updatedItemData = await response.json();

            Alert.alert("Succès", `Prix de "${updatedItemData.product_name}" mis à jour : ${updatedItemData.price} €`);

            handleCloseModal();

        } catch (error) {
            console.error("Erreur lors de la soumission du prix:", error);
            Alert.alert("Erreur", `Impossible de mettre à jour le prix. ${error}`);
        }
    };
    const itemExists = item?.getPrice() !== 0;

    return (
        <View style={styles.container}>
            <CameraView
                style={styles.camera}
                facing={facing}
                onBarcodeScanned={scanned ? undefined : handleBarcodeScanned}
                barcodeScannerSettings={{
                    barcodeTypes: ['ean13', 'qr'],
                }}
            />

            {isLoading && <ActivityIndicator size="large" color="#fff" style={styles.loading} />}

            <Modal
                visible={isModalVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={handleCloseModal}
            >
                <View style={styles.modalContainer}>
                    <View style={styles.modalContent}>
                        {item && (
                            <>
                                <Text style={styles.modalTitle}>
                                    {itemExists ? "Modifier le prix" : "Ajouter un prix"}
                                </Text>

                                <Image
                                    style={styles.itemImage}
                                    source={{ uri: item.getImageSmallUrl() as string }}
                                    resizeMode="contain"
                                />
                                <Text style={styles.itemTextName}>{item.getProductName()}</Text>
                                <Text style={styles.itemText}>
                                    {`${item.getProductQuantity()} ${item.getProductQuantityUnit()}`}
                                </Text>

                                {itemExists && (
                                    <Text style={styles.itemTextOldPrice}>
                                        Ancien prix : {`${item.getPrice()}`} €
                                    </Text>
                                )}

                                <TextInput
                                    style={styles.input}
                                    placeholder="Nouveau prix (€)"
                                    keyboardType="numeric"
                                    value={newPrice}
                                    onChangeText={setNewPrice}
                                />

                                <View style={styles.buttonContainer}>
                                    <Button title="Annuler" onPress={handleCloseModal} color="red" />
                                    <Button title="Valider" onPress={handlePriceSubmit} />
                                </View>
                            </>
                        )}
                    </View>
                </View>
            </Modal>
        </View>
    );
}

const styles = StyleSheet.create({
    container: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: '#000',
    },
    permissionText: {
        color: 'white',
        textAlign: 'center',
        margin: 20,
    },
    camera: {
        ...StyleSheet.absoluteFillObject,
    },
    loading: {
        position: 'absolute',
    },
    modalContainer: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: 'rgba(0,0,0,0.5)',
    },
    modalContent: {
        width: '90%',
        backgroundColor: 'white',
        borderRadius: 10,
        padding: 20,
        alignItems: 'center',
        shadowColor: '#000',
        shadowOffset: {
            width: 0,
            height: 2,
        },
        shadowOpacity: 0.25,
        shadowRadius: 4,
        elevation: 5,
    },
    modalTitle: {
        fontSize: 20,
        fontWeight: 'bold',
        marginBottom: 15,
    },
    itemImage: {
        width: 100,
        height: 100,
        marginBottom: 10,
    },
    itemTextName: {
        fontSize: 16,
        fontWeight: 'bold',
        textAlign: 'center',
    },
    itemText: {
        fontSize: 14,
        color: 'gray',
        marginBottom: 10,
    },
    itemTextOldPrice: {
        fontSize: 16,
        color: 'blue',
        marginBottom: 10,
    },
    input: {
        width: '80%',
        height: 40,
        borderColor: 'gray',
        borderWidth: 1,
        borderRadius: 5,
        paddingHorizontal: 10,
        marginBottom: 20,
        textAlign: 'center',
    },
    buttonContainer: {
        flexDirection: 'row',
        justifyContent: 'space-around',
        width: '100%',
    }
});