import React, { useState, useEffect, useCallback } from 'react';
import { View, Text, ScrollView, StyleSheet, Image, TouchableOpacity, ActivityIndicator } from 'react-native';
import { FontAwesome, MaterialCommunityIcons } from '@expo/vector-icons';
import { CartItem } from '../types/CartItem';
import { CartService } from '../services/CartService';
import { useNavigation } from '@react-navigation/native';

interface CartItemRowProps {
    cartItem: CartItem;
    onQuantityChange: () => void;
}

interface CartScreenProps {
    onValidateCart?: () => void;
}

const CartItemRow: React.FC<CartItemRowProps> = ({ cartItem, onQuantityChange }) => {
    const item = cartItem.getItem();
    const quantity = cartItem.getQuantity();
    const totalPrice = cartItem.getTotalPrice();

    const handleDecrement = () => {
        CartService.decrementItemQuantity(item.getBarcodeId());
        onQuantityChange();
    };
    const handleIncrement = () => {
        CartService.incrementItemQuantity(item.getBarcodeId());
        onQuantityChange();
    };
    const handleRemove = () => {
        CartService.removeItemFromCart(item.getBarcodeId());
        onQuantityChange();
    };

    return (
        <View style={styles.card}>
            <Image
                source={{ uri: item.getSmallImageUrl() as string }}
                style={styles.itemImage}
                resizeMode="contain"
            />
            <View style={styles.cardContent}>
                <View style={styles.cardHeader}>
                    <Text style={styles.itemName} numberOfLines={2}>{item.getProductName()}</Text>
                    <TouchableOpacity onPress={handleRemove} style={styles.deleteButton}>
                        <MaterialCommunityIcons name="trash-can-outline" size={22} color="#FF3B30" />
                    </TouchableOpacity>
                </View>

                <Text style={styles.unitPrice}>PU: {(item.getPrice() as number).toFixed(2)} €</Text>

                <View style={styles.cardFooter}>
                    <View style={styles.quantityContainer}>
                        <TouchableOpacity onPress={handleDecrement} style={styles.qtyBtn}>
                            <FontAwesome name="minus" size={12} color="#333" />
                        </TouchableOpacity>
                        <Text style={styles.qtyText}>{quantity}</Text>
                        <TouchableOpacity onPress={handleIncrement} style={styles.qtyBtn}>
                            <FontAwesome name="plus" size={12} color="#333" />
                        </TouchableOpacity>
                    </View>
                    <Text style={styles.rowTotal}>{(totalPrice as number).toFixed(2)} €</Text>
                </View>
            </View>
        </View>
    );
};

const CartScreen: React.FC<CartScreenProps> = ({ onValidateCart }) => {
    const [cartItems, setCartItems] = useState<CartItem[]>([]);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const navigation = useNavigation<any>();

    const refreshCart = useCallback(() => {
        setCartItems(CartService.getCartItems());
    }, []);

    const formatPrice = (price: number): string => {
        return new Intl.NumberFormat('fr-FR', {
            style: 'currency',
            currency: 'EUR',
        }).format(price);
    };

    useEffect(() => {
        const initCart = async () => {
            setIsLoading(true);
            try {
                await CartService.fetchCartFromServer();
                refreshCart();
            } catch (error) {
                console.error("Impossible de récupérer le panier serveur", error);
            } finally {
                setIsLoading(false);
            }
        };

        initCart();
    }, [refreshCart]);

    const grandTotal = CartService.getTotalPrice() as number;
    const distinctCount = CartService.getDistinctItemCount();

    if (isLoading) {
        return (
            <View style={[styles.container, styles.loadingContainer]}>
                <ActivityIndicator size="large" color="#007AFF" />
                <Text style={styles.loadingText}>Chargement du panier...</Text>
            </View>
        );
    }

    return (
        <View style={styles.container}>
            <View style={styles.headerContainer}>
                <Text style={styles.headerTitle}>Mon Panier</Text>
            </View>

            <ScrollView style={styles.scrollView} contentContainerStyle={{ paddingBottom: 100 }}>
                {cartItems.length === 0 ? (
                    <View style={styles.emptyContainer}>
                        <MaterialCommunityIcons name="cart-off" size={60} color="#ccc" />
                        <Text style={styles.emptyText}>Votre panier est vide</Text>
                    </View>
                ) : (
                    cartItems.map((cartItem) => (
                        <CartItemRow
                            key={cartItem.getItem().getBarcodeId() as string}
                            cartItem={cartItem}
                            onQuantityChange={refreshCart}
                        />
                    ))
                )}
            </ScrollView>

            {distinctCount > 0 && (
                <View style={styles.footer}>
                    <Text style={styles.qtyText}>Articles distincts: {distinctCount}</Text>
                    <Text style={styles.rowTotal}>Total du panier: {formatPrice(grandTotal)}</Text>

                    <TouchableOpacity
                        style={styles.validateButton}
                        onPress={onValidateCart}
                    >
                        <Text style={styles.validateButtonText}>Valider le panier</Text>
                        <FontAwesome name="arrow-right" size={16} color="#fff" style={{ marginLeft: 10 }} />
                    </TouchableOpacity>
                </View>
            )}
        </View>
    );
};

const styles = StyleSheet.create({
    container: {
        flex: 1,
        backgroundColor: '#F2F2F7'
    },

    loadingContainer: {
        justifyContent: 'center',
        alignItems: 'center',
    },

    loadingText: {
        marginTop: 10,
        color: '#8E8E93',
        fontSize: 16
    },

    headerContainer: {
        padding: 20,
        paddingTop: 60,
        backgroundColor: '#FFF',
        borderBottomWidth: 1,
        borderColor: '#E5E5EA'
    },

    headerTitle: {
        fontSize: 28,
        fontWeight: '800',
        color: '#000'
    },

    scrollView: {
        flex: 1,
        padding: 15
    },

    card: {
        flexDirection: 'row',
        backgroundColor: '#FFF',
        borderRadius: 12,
        padding: 12,
        marginBottom: 12,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.1,
        shadowRadius: 4,
        elevation: 3,
    },

    itemImage: {
        width: 70,
        height: 70,
        borderRadius: 8,
        backgroundColor: '#F2F2F7'
    },

    cardContent: {
        flex: 1,
        marginLeft: 12,
        justifyContent: 'space-between'
    },

    cardHeader: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'flex-start'
    },

    itemName: {
        fontSize: 16,
        fontWeight: '600',
        color: '#1C1C1E',
        flex: 1,
        marginRight: 8
    },

    deleteButton: {
        padding: 4
    },

    unitPrice: {
        fontSize: 12,
        color: '#8E8E93',
        marginTop: 4
    },

    cardFooter: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'center',
        marginTop: 8
    },

    quantityContainer: {
        flexDirection: 'row',
        alignItems: 'center',
        backgroundColor: '#F2F2F7',
        borderRadius: 8
    },

    qtyBtn: {
        padding: 8,
        alignItems: 'center',
        justifyContent: 'center',
        width: 32
    },

    qtyText: {
        fontSize: 14,
        fontWeight: '600',
        paddingHorizontal: 8
    },

    rowTotal: {
        fontSize: 16,
        fontWeight: '700',
        color: '#007AFF'
    },

    emptyContainer: {
        alignItems: 'center',
        justifyContent: 'center',
        marginTop: 100
    },

    emptyText: {
        marginTop: 16,
        fontSize: 18,
        color: '#8E8E93'
    },

    footer: {
        position: 'absolute',
        bottom: 0,
        left: 0,
        right: 0,
        backgroundColor: '#FFF',
        padding: 20,
        paddingBottom: 30,
        borderTopWidth: 1,
        borderColor: '#E5E5EA',
        shadowColor: "#000",
        shadowOffset: {
            width: 0,
            height: -3
        },
        shadowOpacity: 0.1,
        shadowRadius: 3
    },

    totalRow: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginBottom: 16
    },

    totalLabel: {
        fontSize: 18,
        color: '#1C1C1E'
    },

    totalAmount: {
        fontSize: 24,
        fontWeight: '800',
        color: '#000'
    },

    checkoutButton: {
        backgroundColor: '#007AFF',
        borderRadius: 14,
        padding: 16,
        flexDirection: 'row',
        justifyContent: 'center',
        alignItems: 'center'
    },

    checkoutButtonText: {
        color: '#FFF',
        fontSize: 17,
        fontWeight: '700',
        marginRight: 8
    },

    validateButton: {
        marginTop: 15,
        backgroundColor: '#007AFF',
        paddingVertical: 12,
        paddingHorizontal: 20,
        borderRadius: 25,
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'center',
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.2,
        shadowRadius: 2,
        elevation: 3,
    },
    validateButtonText: {
        color: '#fff',
        fontSize: 18,
        fontWeight: 'bold',
    },
});

export default CartScreen;