from typing import List, Optional

from sqlalchemy.orm import Session

from ..exceptions import ConditionException, NotFoundException
import httpx
from ..database import engine
from .models import Item
from .schemas import AddItemSchema


def add_item(item: AddItemSchema) -> Optional[Item]:
    with Session(engine) as db:
        if item.price < 100:
            raise ConditionException(detail="Price must be greater or equal than 100")

        db_item: Item = Item(**item.dict())

        db.add(db_item)
        db.commit()
        db.refresh(db_item)

        return db_item


def delete_item(item_id: int) -> Optional[Item]:
    with Session(engine) as db:
        item: Optional[Item] = db.query(
            Item).filter(Item.id == item_id).first()

        if not item:
            raise NotFoundException("Item not found")

        db.delete(item)
        db.commit()

        return item


def get_item(item_id: int) -> Optional[Item]:
    with Session(engine) as db:
        item: Optional[Item] = db.query(
            Item).filter(Item.id == item_id).first()

        if not item:
            raise NotFoundException("Item not found")

        return item


def get_items(offset: int, limit: int) -> List[Item]:
    with Session(engine) as db:
        return db.query(Item).offset(offset).limit(limit).all()

def get_or_create_item_by_barcode(barcode_id: str) -> Optional[Item]:
    """
    Tente de récupérer un item par son code-barres.
    S'il n'existe pas, le crée en appelant Open Food Facts.
    """
    with Session(engine) as db:
        item_db = db.query(Item).filter(Item.barcode_id == barcode_id).first()
        if item_db:
            return item_db

        OPEN_FOOD_FACTS_URL = f"https://world.openfoodfacts.org/api/v0/product/{barcode_id}.json"
        
        try:
            with httpx.Client() as client:
                response = client.get(OPEN_FOOD_FACTS_URL)
                response.raise_for_status() # Lève une exception si erreur HTTP
                data = response.json()

            if data.get("status") == 0 or not data.get("product"):
                raise NotFoundException(detail="Produit non trouvé sur Open Food Facts")

            product_data = data["product"]
            
            new_item = Item(
                barcode_id=barcode_id,
                product_name=product_data.get("product_name", "Inconnu"),
                image_url=product_data.get("image_url"),
                small_image_url=product_data.get("image_small_url"),
                brands=product_data.get("brands"),
                nutrition_grade=product_data.get("nutriscore_grade"),
                product_quantity=product_data.get("product_quantity"),
                product_quantity_unit=product_data.get("product_quantity_unit"),
                ingredients=product_data.get("ingredients_text"),
                allergens=product_data.get("allergens_imported"),
                category=product_data.get("categories_tags", [None])[0],
                price=0
            )
            
            db.add(new_item)
            db.commit()
            db.refresh(new_item)
            
            return new_item

        except httpx.HTTPStatusError:
            raise NotFoundException(detail="Échec de la récupération depuis Open Food Facts")
        except Exception as e:
            raise ConditionException(detail=f"Erreur lors du traitement du produit: {str(e)}")