from typing import List
from sqlalchemy.orm import Session
from fastapi import APIRouter, Query, Depends
from . import controllers, schemas
from .controllers import add_item, delete_item, get_item, get_items, get_or_create_item_by_barcode, get_rayons, get_attributs, get_conservations
from .schemas import AddItemSchema, ItemSchema
from src.core.database import get_db

item_router = APIRouter(
    prefix="/items",
    tags=["Items"],
    responses={
        400: {"description": "Bad Request"},
        404: {"description": "Not found"},
    }
)

@item_router.get("/options/rayons", response_model=List[str])
def read_rayons():
    """
    Récupère la liste unique des catégories "Par Rayon"
    depuis le fichier CSV.
    """
    return get_rayons()

@item_router.get("/options/attributs", response_model=List[str])
def read_attributs():
    """
    Récupère la liste unique des catégories "Par Attribut"
    depuis le fichier CSV.
    """
    return get_attributs()


@item_router.get("/options/conservations", response_model=List[str])
def read_conservations():
    """
    Récupère la liste unique des catégories "Par Conservation"
    depuis le fichier CSV.
    """
    return get_conservations()

@item_router.post("/", response_model=ItemSchema)
def create_item(
    item: AddItemSchema,
):
    return add_item(item)


@item_router.get("/{item_id}", response_model=ItemSchema)
def read_item(
    item_id: int,
):
    return get_item(item_id)

@item_router.get("/barcode/{barcode_id}", response_model=ItemSchema)
def read_item_by_barcode(
    barcode_id: str,
):
    return get_or_create_item_by_barcode(barcode_id)

@item_router.get("/", response_model=List[ItemSchema])
def read_items(
    offset: int = 0,
    limit: int = Query(default=100, lte=100),
):
    return get_items(offset, limit)


@item_router.delete("/", response_model=ItemSchema)
def remove_item(
    item_id: int,
):
    return delete_item(item_id)

@item_router.patch("/barcode/price/{barcode_id}", response_model=ItemSchema)
def update_item_price_route(
    barcode_id: str, 
    price_update: schemas.ItemPriceUpdate,
    db: Session = Depends(get_db)
):
    return controllers.update_item_price(
        db=db, 
        barcode_id=barcode_id, 
        price_update=price_update
    )

@item_router.patch("/barcode/details/{barcode_id}", response_model=ItemSchema)
def update_item_details_route(
    barcode_id: str,
    details: schemas.ItemDetailsUpdate,
):
    return controllers.update_item_details(barcode_id, details)