from typing import List, Optional
from sqlalchemy.orm import Session, joinedload

from ..database import engine
from ..exceptions import ConditionException, NotFoundException
from ..customers.models import Customer
from ..items.models import Item
from .models import Cart, CartItem
from .schemas import CartItemCreate

def get_or_create_cart(db: Session, customer_id: str) -> Cart:
    """
    Récupère le panier d'un client. S'il n'existe pas,
    le crée après avoir vérifié que le client existe.
    """
    # Vérifier si le client existe
    customer = db.query(Customer).filter(Customer.id == customer_id).first()
    if not customer:
        raise NotFoundException(detail="Customer not found")

    # Récupérer le panier
    cart = db.query(Cart).filter(Cart.customer_id == customer_id).first()

    # S'il n'existe pas, le créer
    if not cart:
        cart = Cart(customer_id=customer_id)
        db.add(cart)
        db.commit()
        db.refresh(cart)
        
    return cart

def add_item_to_cart(customer_id: str, item_in: CartItemCreate) -> Cart:
    """
    Ajoute un item (ou met à jour sa quantité) au panier d'un client.
    """
    with Session(engine) as db:
        # Récupérer ou créer le panier
        cart = get_or_create_cart(db, customer_id)

        # Vérifier si l'item demandé existe
        item_db = db.query(Item).filter(Item.item_id == item_in.item_id).first()
        if not item_db:
            raise NotFoundException(detail="Item not found")

        # Vérifier si l'item est déjà dans le panier
        cart_item = db.query(CartItem).filter(
            CartItem.cart_id == cart.id,
            CartItem.item_id == item_in.item_id
        ).first()

        if cart_item:
            # Si oui, met à jour la quantité
            cart_item.quantity += item_in.quantity
        else:
            # Si non, crée une nouvelle entrée CartItem
            cart_item = CartItem(
                cart_id=cart.id,
                item_id=item_in.item_id,
                quantity=item_in.quantity
            )
            db.add(cart_item)

        # Valider les changements et rafraîchir
        db.commit()
        db.refresh(cart)
        return get_cart(customer_id) # Retourne le panier à jour

def remove_item_from_cart(customer_id: str, item_id: int) -> Cart:
    """
    Supprime un item du panier d'un client.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db, customer_id)

        # Trouver l'item dans le panier
        cart_item = db.query(CartItem).filter(
            CartItem.cart_id == cart.id,
            CartItem.item_id == item_id
        ).first()

        if not cart_item:
            raise NotFoundException(detail="Item not found in cart")

        # Supprimer l'item
        db.delete(cart_item)
        db.commit()
        
        return get_cart(customer_id) # Retourne le panier à jour

def get_cart(customer_id: str) -> Cart:
    """
    Récupère le contenu complet du panier d'un client.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db, customer_id)
        
        # Recharge le panier avec les relations (items et l'item de l'item)
        # pour éviter les erreurs de session fermée (LazyLoad)
        cart_id = cart.id
        db.expire(cart) # Force le rechargement
        
        cart_full = db.query(Cart).options(
            joinedload(Cart.items).joinedload(CartItem.item)
        ).filter(Cart.id == cart_id).first()

        return cart_full

def clear_cart(customer_id: str) -> Cart:
    """
    Vide entièrement le panier d'un client.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db, customer_id)
        
        # Supprime tous les CartItem liés à ce panier
        db.query(CartItem).filter(CartItem.cart_id == cart.id).delete()
        db.commit()
        
        db.refresh(cart)
        return cart