import React, { useEffect, useState } from 'react';
import { View, FlatList, StyleSheet, RefreshControl, ActivityIndicator, TouchableOpacity } from 'react-native';
import { Text, Card, Divider, Chip, useTheme } from 'react-native-paper';
import { PaymentService, Payment } from '../services/PaiementService';
import { useNavigation } from '@react-navigation/native';
import ScreenTitle from '../components/ScreenTitle';

export default function HistoryScreen() {
    const [payments, setPayments] = useState<Payment[]>([]);
    const [loading, setLoading] = useState<boolean>(true);
    const [refreshing, setRefreshing] = useState<boolean>(false);
    const navigation = useNavigation<any>();
    const theme = useTheme();

    const fetchHistory = async () => {
        try {
            const data = await PaymentService.getAllPayments();
            const sortedData = data.sort((a, b) => {
                const dateA = a.checkout_date ? new Date(a.checkout_date).getTime() : 0;
                const dateB = b.checkout_date ? new Date(b.checkout_date).getTime() : 0;
                return dateB - dateA;
            });
            setPayments(sortedData);
        } catch (error) {
            console.error("Impossible de charger l'historique", error);
        } finally {
            setLoading(false);
            setRefreshing(false);
        }
    };

    useEffect(() => {
        fetchHistory();
    }, []);

    const onRefresh = () => {
        setRefreshing(true);
        fetchHistory();
    };

    const formatDate = (dateString: string | null) => {
        if (!dateString) return 'Date inconnue';
        return new Date(dateString).toLocaleDateString('fr-FR', {
            day: 'numeric', month: 'long', year: 'numeric', hour: '2-digit', minute: '2-digit'
        });
    };

    const renderItem = ({ item }: { item: Payment }) => (
        <TouchableOpacity onPress={() => navigation.navigate('OrderDetailScreen', { payment: item })}>
            <Card style={[styles.card, { backgroundColor: theme.colors.elevation.level1 }]}>
                <Card.Title
                    title={`Commande #${item.id.slice(-6).toUpperCase()}`}
                    subtitle={formatDate(item.checkout_date)}
                    right={(props) => (
                        <Text {...props} style={styles.priceText}>
                            {item.total_price ? `${item.total_price.toFixed(2)} €` : 'N/A'}
                        </Text>
                    )}
                />
                <Card.Content>
                    <View style={styles.statusContainer}>
                        <Chip icon={item.is_checked ? "check" : "clock"} mode="outlined" compact>
                            {item.is_checked ? "Validé" : "En attente"}
                        </Chip>
                        <Text style={[styles.itemCount, { color: theme.colors.onSurfaceVariant }]}>
                            {item.purchased_items.length} article(s)
                        </Text>
                    </View>
                    <Divider style={{ marginVertical: 10 }} />
                    {item.purchased_items.map((pItem, index) => (
                        <Text key={index} numberOfLines={1} style={{ fontSize: 12, color: theme.colors.onSurface }}>
                            - {pItem.amount}x {pItem.item?.product_name || 'Produit inconnu'}
                        </Text>
                    ))}
                </Card.Content>
            </Card>
        </TouchableOpacity>
    );

    if (loading) {
        return (
            <View style={[styles.center, { backgroundColor: theme.colors.background }]}>
                <ActivityIndicator size="large" color={theme.colors.primary} />
                <Text style={{ marginTop: 10, color: theme.colors.onBackground }}>Chargement...</Text>
            </View>
        );
    }

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <ScreenTitle title="Historique des paiements" />
            <FlatList
                data={payments}
                keyExtractor={(item) => item.id}
                renderItem={renderItem}
                contentContainerStyle={styles.listContent}
                refreshControl={
                    <RefreshControl
                        refreshing={refreshing}
                        onRefresh={onRefresh}
                        tintColor={theme.colors.primary}
                        colors={[theme.colors.primary]}
                    />
                }
                ListEmptyComponent={
                    <View style={styles.center}>
                        <Text style={{ color: theme.colors.onSurfaceVariant }}>Aucun paiement trouvé.</Text>
                    </View>
                }
            />
        </View>
    );
}

const styles = StyleSheet.create({
    container: { flex: 1 },
    headerContainer: {
        padding: 20,
        paddingTop: 60,
        borderBottomWidth: 1,
    },
    headerTitle: {
        fontSize: 28,
        fontWeight: '800',
    },
    center: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
    },
    listContent: {
        padding: 16,
        paddingBottom: 80,
    },
    card: {
        marginBottom: 16,
        elevation: 2,
    },
    priceText: {
        fontWeight: 'bold',
        fontSize: 18,
        color: '#22C55E',
        marginRight: 16,
    },
    statusContainer: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'center',
        marginBottom: 5,
    },
    itemCount: {
        fontSize: 12,
    },
});