from fastapi import APIRouter
from typing import List

from .controllers import (
    get_cart, 
    add_item_to_cart, 
    remove_item_from_cart, 
    clear_cart
)
from .schemas import CartSchema, CartItemCreate

cart_router = APIRouter(
    prefix="/cart",
    tags=["Cart"],
    responses={
        400: {"description": "Bad Request"},
        404: {"description": "Not found"},
    }
)

@cart_router.get("/", response_model=CartSchema)
def read_cart():
    """ Récupère le panier (et son contenu) """
    return get_cart()


@cart_router.post("/items", response_model=CartSchema)
def add_item(item: CartItemCreate):
    """ Ajoute un item au panier """
    return add_item_to_cart(item)


@cart_router.delete("/items/{item_id}", response_model=CartSchema)
def remove_item(
    item_id: int,
):
    """ Supprime un item spécifique du panier """
    return remove_item_from_cart(item_id)


@cart_router.delete("/delete", response_model=CartSchema)
def clear(
):
    """ Vide entièrement le panier """
    return clear_cart()