from fastapi import APIRouter
from typing import List

from .controllers import (
    get_cart, 
    add_item_to_cart, 
    remove_item_from_cart, 
    clear_cart
)
from .schemas import CartSchema, CartItemCreate

cart_router = APIRouter(
    prefix="/cart",
    tags=["Cart"],
    responses={
        400: {"description": "Bad Request"},
        404: {"description": "Not found"},
    }
)

@cart_router.get("/{customer_id}", response_model=CartSchema)
def read_cart(
    customer_id: str,
):
    """ Récupère le panier (et son contenu) pour un client """
    return get_cart(customer_id)


@cart_router.post("/{customer_id}/items", response_model=CartSchema)
def add_item(
    customer_id: str,
    item: CartItemCreate,
):
    """ Ajoute un item au panier d'un client """
    return add_item_to_cart(customer_id, item)


@cart_router.delete("/{customer_id}/items/{item_id}", response_model=CartSchema)
def remove_item(
    customer_id: str,
    item_id: int,
):
    """ Supprime un item spécifique du panier d'un client """
    return remove_item_from_cart(customer_id, item_id)


@cart_router.delete("/{customer_id}", response_model=CartSchema)
def clear(
    customer_id: str,
):
    """ Vide entièrement le panier d'un client """
    return clear_cart(customer_id)