from typing import List, Optional, Set
from sqlalchemy.orm import Session
from ..exceptions import ConditionException, NotFoundException
import httpx
from fastapi import HTTPException, status
from ..database import engine
from .models import Item
from .schemas import AddItemSchema
from . import models, schemas
import os
import csv

SRC_DIR = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
BASE_DIR = os.path.dirname(SRC_DIR)
CSV_PATH = os.path.join(BASE_DIR, 'assets', 'categories.csv')

def get_options_from_csv(systeme: str) -> List[str]:
    """
    Lit le fichier assets/categories.csv et extrait les catégories principales
    uniques pour un système de classification donné.
    """
    options: Set[str] = set()
    try:
        with open(CSV_PATH, mode='r', encoding='utf-8') as file:
            reader = csv.DictReader(file, delimiter=';') 
            
            for row in reader:
                if row['SYSTEME_CLASSIFICATION'] == systeme:
                    options.add(row['CATEGORIE_PRINCIPALE'])
                    
    except FileNotFoundError:
        print(f"ERREUR: Fichier CSV non trouvé. Vérifiez : {CSV_PATH}")
        return []
    except Exception as e:
        print(f"ERREUR lors du parsing CSV: {e}")
        return []
    
    return sorted(list(options))

def get_rayons() -> List[str]:
    return get_options_from_csv('Par Rayon')

def get_moments() -> List[str]:
    return get_options_from_csv('Par Moment')

def get_attributs() -> List[str]:
    return get_options_from_csv('Par Attribut')

def get_conservations() -> List[str]:
    return get_options_from_csv('Par Conservation')

def add_item(item: AddItemSchema) -> Optional[Item]:
    with Session(engine) as db:
        if item.price <= 0:
            raise ConditionException(detail="Price must be greater than 0")

        db_item: Item = Item(**item.dict())

        db.add(db_item)
        db.commit()
        db.refresh(db_item)

        return db_item

def delete_item(item_id: int) -> Optional[Item]:
    with Session(engine) as db:
        item: Optional[Item] = db.query(
            Item).filter(Item.item_id == item_id).first()

        if not item:
            raise NotFoundException("Item not found")

        db.delete(item)
        db.commit()

        return item

def get_item(item_id: int) -> Optional[Item]:
    with Session(engine) as db:
        item: Optional[Item] = db.query(
            Item).filter(Item.item_id == item_id).first()

        if not item:
            raise NotFoundException("Item not found")

        return item

def get_item_by_barcode(db: Session, barcode_id: str):
    db_item = db.query(models.Item).filter(models.Item.barcode_id == barcode_id).first()
    if db_item is None:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Item not found")
    return db_item

def get_items(offset: int, limit: int) -> List[Item]:
    with Session(engine) as db:
        return db.query(Item).offset(offset).limit(limit).all()

def get_or_create_item_by_barcode(barcode_id: str) -> Optional[Item]:
    """
    Tente de récupérer un item par son code-barres.
    S'il n'existe pas, le crée en appelant Open Food Facts.
    """
    with Session(engine) as db:
        item_db = db.query(Item).filter(Item.barcode_id == barcode_id).first()
        if item_db:
            return item_db

        OPEN_FOOD_FACTS_URL = f"https://world.openfoodfacts.org/api/v0/product/{barcode_id}.json"
        
        try:
            with httpx.Client() as client:
                response = client.get(OPEN_FOOD_FACTS_URL)
                response.raise_for_status()
                data = response.json()

            if data.get("status") == 0 or not data.get("product"):
                raise NotFoundException(detail="Produit non trouvé sur Open Food Facts")

            product_data = data["product"]

            new_item = Item(
                barcode_id=barcode_id,
                product_name=product_data.get("product_name", "Inconnu"),
                image_url=product_data.get("image_url"),
                small_image_url=product_data.get("image_small_url"),
                brands=product_data.get("brands"),
                nutrition_grade=product_data.get("nutriscore_grade"),
                product_quantity=product_data.get("product_quantity"),
                product_quantity_unit=product_data.get("product_quantity_unit"),
                ingredients=product_data.get("ingredients_text"),
                allergens=product_data.get("allergens_imported"),
                price=0,
                rayon='Autre',
                moment='À tous les repas',
                attribut='Aucun',
                conservation='Sec'
            )
            
            db.add(new_item)
            db.commit()
            db.refresh(new_item)
            
            return new_item

        except httpx.HTTPStatusError:
            raise NotFoundException(detail="Échec de la récupération depuis Open Food Facts")
        except Exception as e:
            raise ConditionException(detail=f"Erreur lors du traitement du produit: {str(e)}")

def update_item_details(barcode_id: str, details: schemas.ItemDetailsUpdate) -> Optional[Item]:
    """
    Met à jour un item en bloc (prix, rayon, moment, attribut, conservation).
    Applique tous les champs reçus (y compris null).
    """
    with Session(engine) as db:
        db_item = db.query(Item).filter(Item.barcode_id == barcode_id).first()
        if not db_item:
            raise NotFoundException("Item not found")

        db_item.price = details.price
        db_item.rayon = details.rayon
        db_item.moment = details.moment
        db_item.attribut = details.attribut
        db_item.conservation = details.conservation
        
        db.commit()
        db.refresh(db_item)
        return db_item

def update_item_price(db: Session, barcode_id: str, price_update: schemas.ItemPriceUpdate):
    db_item = get_item_by_barcode(db=db, barcode_id=barcode_id)
    db_item.price = price_update.price
    db.commit()
    db.refresh(db_item)
    return db_item