import { CameraView, useCameraPermissions, BarcodeScanningResult } from 'expo-camera';
import React, { useState, useEffect } from 'react';
import {
    Button,
    StyleSheet,
    Text,
    View,
    Modal,
    TextInput,
    Image,
    ActivityIndicator,
    Alert,
    ScrollView,
    TouchableOpacity
} from 'react-native';
import { API_BASE_URL } from '../config';
import AppPicker from '../components/AppPicker';
import ScannedItem from '../interfaces/ScannedItem';

export default function ScannerScreen() {
    const [permission, requestPermission] = useCameraPermissions();
    const [scanned, setScanned] = useState(false);
    const [isLoading, setIsLoading] = useState(false);

    const [item, setItem] = useState<ScannedItem | null>(null);

    const [isModalVisible, setIsModalVisible] = useState(false);
    const [isScanningActive, setIsScanningActive] = useState(false);
    const [isNewItem, setIsNewItem] = useState(false);

    const [allRayons, setAllRayons] = useState<string[]>([]);
    const [allMoments, setAllMoments] = useState<string[]>([]);
    const [allAttributs, setAllAttributs] = useState<string[]>([]);
    const [allConservations, setAllConservations] = useState<string[]>([]);

    const [newPrice, setNewPrice] = useState('');
    const [selectedRayon, setSelectedRayon] = useState<string | undefined>();
    const [selectedMoment, setSelectedMoment] = useState<string | undefined>();
    const [selectedAttribut, setSelectedAttribut] = useState<string | undefined>();
    const [selectedConservation, setSelectedConservation] = useState<string | undefined>();

    const fetchOptions = async (
        endpoint: string,
        setter: React.Dispatch<React.SetStateAction<string[]>>
    ) => {
        try {
            const response = await fetch(`${API_BASE_URL}/items/options/${endpoint}`);
            if (!response.ok) throw new Error(`Erreur réseau pour ${endpoint}`);
            const data: string[] = await response.json();
            setter(data);
        } catch (error) {
            console.error(`Erreur fetchOptions (${endpoint}):`, error);
        }
    };

    useEffect(() => {
        fetchOptions('rayons', setAllRayons);
        fetchOptions('moments', setAllMoments);
        fetchOptions('attributs', setAllAttributs);
        fetchOptions('conservations', setAllConservations);
    }, []);

    if (!permission) {
        return (
            <View style={styles.container}>
                <ActivityIndicator size="large" color="#fff" />
            </View>
        );
    }

    if (!permission.granted) {
        return (
            <View style={styles.container}>
                <Text style={styles.permissionText}>Nous avons besoin de votre permission pour utiliser la caméra</Text>
                <Button onPress={requestPermission} title="Accorder la permission" />
            </View>
        );
    }

    const handleBarcodeScanned = async (scanningResult: BarcodeScanningResult) => {
        if (scanned) return;
        setIsScanningActive(false);
        setScanned(true);
        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/${scanningResult.data}`);
            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Produit non trouvé");
            }

            const fetchedItem: ScannedItem = await response.json();
            setItem(fetchedItem);

            const isNew = fetchedItem.rayon === null || fetchedItem.price === 0;
            setIsNewItem(isNew);

            setNewPrice(fetchedItem.price ? fetchedItem.price.toString() : '');

            setSelectedRayon(fetchedItem.rayon || undefined);
            setSelectedMoment(fetchedItem.moment || undefined);
            setSelectedAttribut(fetchedItem.attribut || undefined);
            setSelectedConservation(fetchedItem.conservation || undefined);

            setIsModalVisible(true);

        } catch (error) {
            console.error("Erreur handleBarcodeScanned:", error);
            Alert.alert("Erreur", `Impossible de récupérer les détails pour ${scanningResult.data}. ${error}`);
            setScanned(false);
        } finally {
            setIsLoading(false);
        }
    };

    const handleCloseModal = () => {
        setIsModalVisible(false);
        setItem(null);
        setScanned(false);
        setNewPrice('');
        setSelectedRayon(undefined);
        setSelectedMoment(undefined);
        setSelectedAttribut(undefined);
        setSelectedConservation(undefined);
    };

    const handleSubmitUpdates = async () => {
        if (!item) return;
        const priceValue = parseFloat(newPrice.replace(',', '.'));
        if (isNaN(priceValue) || priceValue <= 0) {
            Alert.alert("Erreur", "Le prix doit être un nombre valide et supérieur à 0.");
            return;
        }

        if (!selectedRayon) {
            Alert.alert("Erreur", "Le 'Rayon' est obligatoire.");
            return;
        }
        if (!selectedMoment) {
            Alert.alert("Erreur", "Le 'Moment' est obligatoire.");
            return;
        }
        if (!selectedAttribut) {
            Alert.alert("Erreur", "L''Attribut' est obligatoire.");
            return;
        }
        if (!selectedConservation) {
            Alert.alert("Erreur", "La 'Conservation' est obligatoire.");
            return;
        }

        const updateDetails = {
            price: priceValue,
            rayon: selectedRayon,
            moment: selectedMoment,
            attribut: selectedAttribut,
            conservation: selectedConservation,
        };

        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/details/${item.barcode_id}`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(updateDetails),
            });

            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Erreur lors de la mise à jour");
            }

            Alert.alert("Succès", "Article mis à jour !");
            handleCloseModal();

        } catch (error) {
            console.error("Erreur handleSubmitUpdates:", error);
            Alert.alert("Erreur", `Échec de la mise à jour: ${error}`);
        } finally {
            setIsLoading(false);
        }
    };

    const renderStartButton = () => {
        if (!isScanningActive && !isModalVisible && !isLoading) {
            return (
                <View style={styles.startScanButtonWrapper}>
                    <TouchableOpacity
                        style={styles.startScanButton}
                        onPress={() => setIsScanningActive(true)}
                    >
                        <Text style={styles.startScanButtonText}>Commencer le scan</Text>
                    </TouchableOpacity>
                </View>
            );
        }
        return null;
    };

    return (
        <View style={styles.container}>
            <CameraView
                style={styles.camera}
                facing="back"
                onBarcodeScanned={isScanningActive && !scanned ? handleBarcodeScanned : undefined}
                barcodeScannerSettings={{
                    barcodeTypes: ['ean13', 'qr'],
                }}
            />

            {isLoading && !isModalVisible &&
                <ActivityIndicator size="large" color="#fff" style={styles.loading} />
            }

            {renderStartButton()}

            <Modal
                visible={isModalVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={handleCloseModal}
            >
                <View style={styles.modalContainer}>

                    <View style={styles.modalView}>

                        <ScrollView contentContainerStyle={styles.modalScrollContent}>
                            {item && (
                                <>
                                    <Text style={styles.modalTitle}>
                                        {isNewItem ? "Nouvel Article" : "Mettre à jour l'article"}
                                    </Text>

                                    <Image
                                        style={styles.itemImage}
                                        source={{ uri: item.small_image_url || undefined }}
                                        resizeMode="contain"
                                    />
                                    <Text style={styles.itemTextName}>{item.product_name}</Text>

                                    <Text style={styles.label}>Prix (€)</Text>
                                    <TextInput
                                        style={styles.input}
                                        keyboardType="numeric"
                                        value={newPrice}
                                        onChangeText={setNewPrice}
                                        placeholder="0.00"
                                        placeholderTextColor="#999"
                                    />

                                    <AppPicker
                                        label="Rayon"
                                        selectedValue={selectedRayon}
                                        onValueChange={setSelectedRayon}
                                        items={allRayons}
                                    />

                                    <AppPicker
                                        label="Moment"
                                        selectedValue={selectedMoment}
                                        onValueChange={setSelectedMoment}
                                        items={allMoments}
                                    />

                                    <AppPicker
                                        label="Attribut"
                                        selectedValue={selectedAttribut}
                                        onValueChange={setSelectedAttribut}
                                        items={allAttributs}
                                    />

                                    <AppPicker
                                        label="Conservation"
                                        selectedValue={selectedConservation}
                                        onValueChange={setSelectedConservation}
                                        items={allConservations}
                                    />

                                    <View style={styles.buttonContainer}>
                                        <Button title="Annuler" onPress={handleCloseModal} color="red" />

                                        {isLoading && isModalVisible ? (
                                            <ActivityIndicator style={{ paddingHorizontal: 20 }} />
                                        ) : (
                                            <Button title="Valider" onPress={handleSubmitUpdates} />
                                        )}
                                    </View>
                                </>
                            )}
                        </ScrollView>
                    </View>
                </View>
            </Modal>
        </View>
    );
}


const styles = StyleSheet.create({
    startScanButtonWrapper: {
        position: 'absolute',
        bottom: 40,
        width: '80%',
        alignSelf: 'center',
    },
    startScanButton: {
        backgroundColor: '#007bff',
        paddingVertical: 15,
        borderRadius: 25,
        alignItems: 'center',
        justifyContent: 'center',
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 4 },
        shadowOpacity: 0.3,
        shadowRadius: 5,
        elevation: 8,
    },
    startScanButtonText: {
        color: 'white',
        fontSize: 18,
        fontWeight: 'bold',
    },
    container: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: '#000',
    },
    permissionText: {
        color: 'white',
        textAlign: 'center',
        margin: 20,
    },
    camera: {
        ...StyleSheet.absoluteFillObject,
    },
    loading: {
        position: 'absolute',
    },

    modalContainer: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: 'rgba(0,0,0,0.6)',
    },
    modalView: {
        width: '90%',
        maxHeight: '85%',
        backgroundColor: 'white',
        borderRadius: 20,
        padding: 20,
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.25,
        shadowRadius: 4,
        elevation: 5,
    },
    modalScrollContent: {
        alignItems: 'center',
    },

    modalTitle: {
        fontSize: 22,
        fontWeight: 'bold',
        marginBottom: 15,
        color: '#111',
    },
    itemImage: {
        width: 150,
        height: 150,
        marginBottom: 15,
        borderRadius: 8,
    },
    itemTextName: {
        fontSize: 18,
        fontWeight: '600',
        textAlign: 'center',
        marginBottom: 20,
    },

    label: {
        fontWeight: 'bold',
        marginTop: 10,
        color: '#333',
        width: '100%'
    },
    input: {
        width: '100%',
        height: 50,
        borderColor: '#ccc',
        borderWidth: 1,
        borderRadius: 8,
        paddingHorizontal: 15,
        marginBottom: 10,
        marginTop: 5,
        backgroundColor: '#f9f9f9',
        fontSize: 16,
    },

    buttonContainer: {
        flexDirection: 'row',
        justifyContent: 'space-around',
        width: '100%',
        marginTop: 25,
        marginBottom: 10,
    }
});