import React, { useEffect, useState } from 'react';
import { View, Text, StyleSheet, ScrollView, Alert, ActivityIndicator } from 'react-native';
import { Button, Divider } from 'react-native-paper';
import { CartService } from '../services/CartService';
import { CartItem } from '../types/CartItem';
import { useStripe } from "@stripe/stripe-react-native";
import { API_BASE_URL } from "../config";

export default function RecapScreen() {
    const [cartItems, setCartItems] = useState<CartItem[]>([]);
    const { initPaymentSheet, presentPaymentSheet } = useStripe();
    const [loading, setLoading] = useState(false);
    const [paymentReady, setPaymentReady] = useState(false);

    const userId = "cus_THVVhujjj328BN";

    useEffect(() => {
        setCartItems(CartService.getCartItems());
        initializePaymentSheet();
    }, []);

    const subTotal = CartService.getTotalPrice() as number;
    const tvaRate = 0.20;
    const tvaAmount = subTotal * tvaRate;
    const totalToPay = subTotal + tvaAmount;


    const fetchPaymentSheetParams = async () => {
        const pendingItems = CartService.getCartItems().map(ci => ({
            id: ci.getItem().getBarcodeId(),
            amount: ci.getQuantity()
        }));

        if (pendingItems.length === 0) return null;

        try {
            const response = await fetch(`${API_BASE_URL}/payments/`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    "pending_items": pendingItems,
                    "customer_id": userId
                })
            });

            const { paymentIntent, ephemeralKey, customer } = await response.json();
            return { paymentIntent, ephemeralKey, customer };
        } catch (error) {
            console.error("Erreur fetch params", error);
            Alert.alert("Erreur", "Impossible de contacter le serveur de paiement");
            return null;
        }
    };

    const initializePaymentSheet = async () => {
        setLoading(true);
        const params = await fetchPaymentSheetParams();

        if (!params) {
            setLoading(false);
            return;
        }

        const { paymentIntent, ephemeralKey, customer } = params;

        const { error } = await initPaymentSheet({
            merchantDisplayName: "Mon Magasin",
            customerId: customer,
            customerEphemeralKeySecret: ephemeralKey,
            paymentIntentClientSecret: paymentIntent,
            allowsDelayedPaymentMethods: false,
        });

        if (!error) {
            setPaymentReady(true);
        }
        setLoading(false);
    };

    const handlePayment = async () => {
        const { error } = await presentPaymentSheet();

        if (error) {
            Alert.alert(`Erreur code: ${error.code}`, error.message);
        } else {
            Alert.alert('Succès', 'Votre commande est confirmée !');
            CartService.clearCart();
            setCartItems([]);
            setPaymentReady(false);
        }
    };


    return (
        <View style={styles.container}>
            <ScrollView style={styles.scrollContainer}>
                <Text style={styles.title}>Facture</Text>
                <View style={styles.invoiceContainer}>

                    <View style={styles.tableHeader}>
                        <Text style={[styles.colProduct, styles.headerText]}>Produit</Text>
                        <Text style={[styles.colQty, styles.headerText]}>Qté</Text>
                        <Text style={[styles.colPrice, styles.headerText]}>Total</Text>
                    </View>
                    <Divider />

                    {cartItems.map((ci, index) => (
                        <View key={index} style={styles.tableRow}>
                            <Text style={styles.colProduct} numberOfLines={1}>
                                {ci.getItem().getProductName()}
                            </Text>
                            <Text style={styles.colQty}>x{ci.getQuantity()}</Text>
                            <Text style={styles.colPrice}>
                                {(ci.getTotalPrice() as number).toFixed(2)} €
                            </Text>
                        </View>
                    ))}

                    <Divider style={{ marginVertical: 10 }} />

                    <View style={styles.summaryRow}>
                        <Text style={styles.summaryLabel}>Sous-total (HT) :</Text>
                        <Text style={styles.summaryValue}>{subTotal.toFixed(2)} €</Text>
                    </View>

                    <View style={styles.summaryRow}>
                        <Text style={styles.summaryLabel}>TVA (20%) :</Text>
                        <Text style={styles.summaryValue}>{tvaAmount.toFixed(2)} €</Text>
                    </View>

                    <View style={[styles.summaryRow, styles.totalRow]}>
                        <Text style={styles.totalLabel}>TOTAL À PAYER :</Text>
                        <Text style={styles.totalValue}>{totalToPay.toFixed(2)} €</Text>
                    </View>
                </View>
            </ScrollView>

            <View style={styles.footer}>
                {loading ? (
                    <ActivityIndicator size="large" color="#007AFF" />
                ) : (
                    <Button
                        mode="contained"
                        onPress={handlePayment}
                        disabled={!paymentReady || cartItems.length === 0}
                        style={styles.payButton}
                        contentStyle={{ height: 50 }}
                        labelStyle={{ fontSize: 18, fontWeight: 'bold' }}
                    >
                        Payer {totalToPay.toFixed(2)} €
                    </Button>
                )}
                <Text style={styles.secureText}>🔒 Paiement sécurisé par Stripe</Text>
            </View>
        </View>
    );
}

const styles = StyleSheet.create({
    container: { flex: 1, backgroundColor: '#F5F5F5' },
    scrollContainer: { flex: 1, padding: 20 },
    title: { fontSize: 24, fontWeight: 'bold', marginBottom: 20, textAlign: 'center', color: '#333' },

    invoiceContainer: {
        backgroundColor: 'white',
        borderRadius: 10,
        padding: 20,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.1,
        shadowRadius: 4,
        elevation: 3,
    },

    // Table Styles
    tableHeader: { flexDirection: 'row', marginBottom: 10 },
    headerText: { fontWeight: '700', color: '#666', fontSize: 12, textTransform: 'uppercase' },
    tableRow: { flexDirection: 'row', paddingVertical: 12, borderBottomWidth: 0.5, borderColor: '#eee' },

    colProduct: { flex: 3, color: '#333', fontSize: 14 },
    colQty: { flex: 1, textAlign: 'center', color: '#666' },
    colPrice: { flex: 1.5, textAlign: 'right', fontWeight: '600', color: '#333' },

    // Summary Styles
    summaryRow: { flexDirection: 'row', justifyContent: 'space-between', marginTop: 8 },
    summaryLabel: { color: '#666', fontSize: 14 },
    summaryValue: { color: '#333', fontSize: 14, fontWeight: '500' },

    totalRow: { marginTop: 15, paddingTop: 15, borderTopWidth: 1, borderTopColor: '#eee' },
    totalLabel: { fontSize: 18, fontWeight: '800', color: '#000' },
    totalValue: { fontSize: 20, fontWeight: '800', color: '#007AFF' },

    // Footer & Button
    footer: { padding: 20, backgroundColor: 'white', borderTopWidth: 1, borderColor: '#eee' },
    payButton: { backgroundColor: '#007AFF', borderRadius: 8 },
    secureText: { textAlign: 'center', marginTop: 10, color: '#888', fontSize: 12 }
});