import { Item } from '../types/Item';
import { CartItem } from '../types/CartItem';
import { API_BASE_URL } from '../config';

/*
Service pour gérer le panier d'achat.
Fonctionnalités:
- Ajout, incrémentation, décrémentation et suppression d'articles.
- Synchronisation avec le serveur via des appels API.
- Gestion locale optimiste pour une meilleure expérience utilisateur.
- Récupération et calcul des totaux du panier.
*/

export class CartService {
    private static cartItems: Map<String, CartItem> = new Map();

    public static async fetchCartFromServer(): Promise<void> {

        try {
            const response = await fetch(`${API_BASE_URL}/cart/`);

            if (!response.ok) {
                throw new Error('Erreur réseau lors de la récupération du panier');
            }

            const data = await response.json();

            this.cartItems.clear();

            if (data.items && Array.isArray(data.items)) {
                data.items.forEach((serverCartItem: any) => {
                    const productData = serverCartItem.item;
                    const item = new Item(
                        productData.item_id,
                        productData.barcode_id,
                        productData.product_name,
                        productData.image_url,
                        productData.small_image_url,
                        productData.brands,
                        productData.nutrition_grade,
                        productData.product_quantity,
                        productData.product_quantity_unit,
                        productData.ingredients,
                        productData.allergens,
                        productData.price,
                        productData.rayon,
                        productData.attribut,
                        productData.conservation
                    );

                    const cartItem = new CartItem(item, serverCartItem.quantity);

                    this.cartItems.set(item.getBarcodeId(), cartItem);
                });
            }

            console.log("Panier synchronisé. Nombre d'articles :", this.cartItems.size);

        } catch (error) {
            console.error("Erreur sync panier:", error);
            throw error;
        }
    }

    /**
         * Ajoute un article au panier.
         * Met à jour localement (optimistic) puis envoie au serveur.
         */
    public static addItemToCart(item: Item): CartItem {
        const barcodeId = item.getBarcodeId();
        let cartItem = CartService.cartItems.get(barcodeId);

        // 1. Mise à jour Locale (Optimistic UI)
        if (cartItem) {
            cartItem.incrementQuantity(1);
        } else {
            cartItem = new CartItem(item, 1);
            CartService.cartItems.set(barcodeId, cartItem);
        }

        // 2. Envoi asynchrone au serveur
        // Note: On suppose que l'objet 'item' possède un moyen d'accéder à son ID SQL (item_id)
        // Si votre classe Item n'a pas l'ID interne, il faudra le gérer.
        // Ici j'utilise item.getId() à titre d'exemple.
        const itemId = (item as any).getId ? (item as any).getId() : item.getId();

        this.syncItemToServer(itemId, 1).catch(err => {
            console.error("Erreur ajout serveur, rollback possible ici", err);
        });

        console.log(`Article ${item.getProductName()} ajouté.`);
        return cartItem;
    }

    /**
     * Incrémente la quantité d'un article.
     */
    public static incrementItemQuantity(barcodeId: string): CartItem | undefined {
        const cartItem = CartService.cartItems.get(barcodeId);
        if (cartItem) {
            // Local
            cartItem.incrementQuantity(1);

            // Serveur
            const item = cartItem.getItem();
            const itemId = (item as any).getId ? (item as any).getId() : (item as any).item_id;

            this.syncItemToServer(itemId, 1);

            return cartItem;
        }
        return undefined;
    }

    /**
     * Décrémente la quantité.
     * Si quantité tombe à 0, supprime l'article (DELETE).
     * Sinon, décrémente (POST avec quantité négative si supporté, ou logique custom).
     * Note: Le backend Python actuel fait `quantity += item_in.quantity`, donc -1 fonctionne.
     */
    public static decrementItemQuantity(barcodeId: string): CartItem | undefined {
        const cartItem = CartService.cartItems.get(barcodeId);

        if (cartItem) {
            const item = cartItem.getItem();
            const itemId = (item as any).getId ? (item as any).getId() : (item as any).item_id;

            // Logique locale
            cartItem.decrementQuantity(1);

            if (cartItem.getQuantity() <= 0) {
                // Suppression locale
                CartService.cartItems.delete(barcodeId);
                // Suppression Serveur
                this.deleteItemFromServer(itemId);
                return undefined;
            } else {
                // Décrémentation Serveur (ajout de -1)
                this.syncItemToServer(itemId, -1);
                return cartItem;
            }
        }
        return undefined;
    }

    /**
     * Supprime totalement un article du panier.
     */
    public static removeItemFromCart(barcodeId: string): boolean {
        const cartItem = CartService.cartItems.get(barcodeId);
        if (cartItem) {
            const item = cartItem.getItem();
            const itemId = (item as any).getId ? (item as any).getId() : (item as any).item_id;

            // Local
            const wasDeleted = CartService.cartItems.delete(barcodeId);

            // Serveur
            this.deleteItemFromServer(itemId);

            return wasDeleted;
        }
        return false;
    }

    /**
     * Vide tout le panier.
     */
    public static clearCart(): void {
        // Local
        CartService.cartItems.clear();

        // Serveur (DELETE /cart/delete)
        fetch(`${API_BASE_URL}/cart/delete`, {
            method: 'DELETE',
        }).catch(err => console.error("Erreur clearCart serveur", err));
    }

    /**
     * Helpers pour l'affichage
     */
    public static getDistinctItemCount(): number {
        return this.cartItems.size;
    }

    public static getCartItems(): CartItem[] {
        return Array.from(CartService.cartItems.values());
    }

    public static getTotalPrice(): number {
        let total: number = 0;
        CartService.cartItems.forEach(cartItem => {
            total += (cartItem.getTotalPrice() as number);
        });
        return total;
    }

    /**
     * Envoie une modification de quantité (+1 ou -1) via POST /cart/items
     */
    private static async syncItemToServer(itemId: number, quantityChange: number): Promise<void> {
        if (!itemId) return; // Sécurité

        try {
            await fetch(`${API_BASE_URL}/cart/items`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    item_id: itemId,
                    quantity: quantityChange
                })
            });
        } catch (error) {
            console.error(`Erreur sync item ${itemId}:`, error);
        }
    }

    /**
     * Supprime un item via DELETE /cart/items/{id}
     */
    private static async deleteItemFromServer(itemId: number): Promise<void> {
        if (!itemId) return;

        try {
            await fetch(`${API_BASE_URL}/cart/items/${itemId}`, {
                method: 'DELETE'
            });
        } catch (error) {
            console.error(`Erreur delete item ${itemId}:`, error);
        }
    }

    // Méthode utilitaire pour mettre à jour un item (ex: scan après coup)
    public static updateCartItem(updatedItem: Item) {
        const barcodeId = updatedItem.getBarcodeId();
        const cartItem = CartService.cartItems.get(barcodeId);

        if (cartItem) {
            cartItem.updateItem(updatedItem);
            return cartItem;
        }
        return undefined;
    }
}