import React, { useEffect, useState } from 'react';
import { View, FlatList, StyleSheet, Image, TouchableOpacity, ActivityIndicator, Alert } from 'react-native';
import { Text, Button, useTheme, IconButton } from 'react-native-paper';
import { useRoute, useNavigation, RouteProp } from '@react-navigation/native';
import { API_BASE_URL } from '../../config';
import { Item } from '../../types/Item';
import { CartService } from '../../services/CartService';
import { SelectionStackParamList } from '../../types/Navigation';
import { StackNavigationProp } from '@react-navigation/stack';
import ErrorNotification from '../../components/ErrorNotification';
import ScreenTitle from '../../components/ScreenTitle';

/*
Un écran pour afficher les articles d'un rayon spécifique.
Fonctionnalités:
- Récupère et affiche les articles depuis l'API en fonction du rayon sélectionné.
- Permet d'ajouter des articles au panier.
- Affiche des notifications de succès ou d'erreur lors de l'ajout au panier.
*/

export default function ItemsByRayonScreen() {
    const route = useRoute<RouteProp<SelectionStackParamList, 'ItemsByRayon'>>();
    const navigation = useNavigation<StackNavigationProp<SelectionStackParamList>>();
    const { rayon } = route.params;
    const theme = useTheme();

    const [items, setItems] = useState<Item[]>([]);
    const [loading, setLoading] = useState(true);
    const [notification, setNotification] = useState<{ msg: string, type: 'error' | 'success' } | null>(null);

    useEffect(() => {
        fetchItems();
    }, []);

    const fetchItems = async () => {
        setLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/?limit=500`);
            if (response.ok) {
                const data = await response.json();
                const allItems = data.map((i: any) => new Item(
                    i.item_id, i.barcode_id, i.product_name, i.image_url, i.small_image_url,
                    i.brands, i.nutrition_grade, i.product_quantity, i.product_quantity_unit,
                    i.ingredients, i.allergens, i.price, i.rayon, i.attribut, i.conservation
                ));

                const filtered = allItems.filter((i: Item) => i.getRayon() === rayon);
                setItems(filtered);
            }
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    const handleAddToCart = (item: Item) => {
        CartService.addItemToCart(item);
        setNotification({ msg: "Ajouté au panier", type: 'success' });
    };

    const renderItem = ({ item }: { item: Item }) => (
        <TouchableOpacity
            style={[styles.itemCard, { backgroundColor: theme.colors.elevation.level1 }]}
            onPress={() => navigation.navigate('ItemDetail', { item, onRefresh: fetchItems })}
        >
            <Image
                source={{ uri: item.getSmallImageUrl() || 'https://via.placeholder.com/100' }}
                style={styles.image}
                resizeMode="contain"
            />
            <View style={styles.infoContainer}>
                <Text variant="titleSmall" numberOfLines={2} style={{ color: theme.colors.onSurface }}>
                    {item.getProductName()}
                </Text>
                <Text variant="bodySmall" style={{ color: theme.colors.primary, fontWeight: 'bold' }}>
                    {(item.getPrice() as number).toFixed(2)} €
                </Text>
                <Button
                    mode="contained-tonal"
                    compact
                    style={{ marginTop: 5 }}
                    onPress={() => handleAddToCart(item)}
                >
                    Ajouter
                </Button>
            </View>
        </TouchableOpacity>
    );

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <ScreenTitle title={`${rayon}`} />

            {loading ? (
                <ActivityIndicator size="large" color={theme.colors.primary} style={styles.center} />
            ) : (
                <FlatList
                    data={items}
                    keyExtractor={(item) => item.getId().toString()}
                    renderItem={renderItem}
                    numColumns={2}
                    contentContainerStyle={styles.list}
                />
            )}
            <ErrorNotification
                message={notification?.msg || null}
                type={notification?.type}
                onClose={() => setNotification(null)}
            />
        </View>
    );
}

const styles = StyleSheet.create({
    container: {
        flex: 1
    },

    center: {
        marginTop: 50
    },

    list: {
        padding: 10
    },

    itemCard: {
        flex: 1,
        margin: 5,
        padding: 10,
        borderRadius: 8,
        alignItems: 'center',
        maxWidth: '48%',
    },

    image: {
        width: 80,
        height: 80,
        marginBottom: 8,
        borderRadius: 4
    },

    infoContainer: {
        width: '100%',
        alignItems: 'center'
    }
});