import React, { useState, useEffect } from 'react';
import { View, ScrollView, StyleSheet, Image, Alert } from 'react-native';
import { TextInput, Button, Text, useTheme, Divider, Surface } from 'react-native-paper';
import { useRoute, useNavigation, RouteProp } from '@react-navigation/native';
import { SelectionStackParamList } from '../../types/Navigation';
import AppPicker from '../../components/AppPicker';
import { API_BASE_URL } from '../../config';
import { CartService } from '../../services/CartService';
import ErrorNotification from '../../components/ErrorNotification';
import ScreenTitle from '../../components/ScreenTitle';

/*
Un écran pour afficher et gérer les détails d'un article spécifique.
Fonctionnalités:
- Affiche les informations détaillées de l'article, y compris l'image, le nom, la marque et le code-barres.
- Permet de modifier le prix, le rayon, l'attribut et la conservation de l'article.
- Charge les options disponibles pour le rayon, l'attribut et la conservation depuis l'API.
- Permet de sauvegarder les modifications, d'ajouter l'article au panier ou de supprimer l'article de la base de données.
- Affiche des notifications d'erreur ou de succès lors des opérations.
*/

export default function ItemDetailScreen() {
    const route = useRoute<RouteProp<SelectionStackParamList, 'ItemDetail'>>();
    const navigation = useNavigation();
    const { item, onRefresh } = route.params;
    const theme = useTheme();

    const [formData, setFormData] = useState({
        price: item.getPrice()?.toString(),
        rayon: item.getRayon(),
        attribut: item.getAttribut(),
        conservation: item.getConservation(),
    });

    const [loading, setLoading] = useState(false);
    const [notification, setNotification] = useState<{ msg: string, type: 'error' | 'success' } | null>(null);

    const [optRayons, setOptRayons] = useState<string[]>([]);
    const [optAttributs, setOptAttributs] = useState<string[]>([]);
    const [optConservations, setOptConservations] = useState<string[]>([]);

    useEffect(() => {
        const loadOptions = async () => {
            try {
                const [r, a, c] = await Promise.all([
                    fetch(`${API_BASE_URL}/items/options/rayons`).then(res => res.json()),
                    fetch(`${API_BASE_URL}/items/options/attributs`).then(res => res.json()),
                    fetch(`${API_BASE_URL}/items/options/conservations`).then(res => res.json())
                ]);
                setOptRayons(r); setOptAttributs(a); setOptConservations(c);
            } catch (e) {
                console.error("Erreur options", e);
            }
        };
        loadOptions();
    }, []);

    const handleUpdate = async () => {
        setLoading(true);
        try {
            const payload = {
                price: parseFloat(formData.price?.replace(',', '.') || '0'),
                rayon: formData.rayon,
                attribut: formData.attribut,
                conservation: formData.conservation,
            };

            const response = await fetch(`${API_BASE_URL}/items/barcode/details/${item.getBarcodeId()}`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });

            if (response.ok) {
                setNotification({ msg: "Produit mis à jour !", type: 'success' });
                if (onRefresh) onRefresh();
            } else {
                throw new Error("Erreur mise à jour");
            }
        } catch (e) {
            setNotification({ msg: "Erreur lors de la mise à jour", type: 'error' });
        } finally {
            setLoading(false);
        }
    };

    const handleDelete = () => {
        Alert.alert(
            "Supprimer",
            "Voulez-vous vraiment supprimer cet article de la base de données ?",
            [
                { text: "Annuler", style: "cancel" },
                {
                    text: "Supprimer",
                    style: "destructive",
                    onPress: async () => {
                        setLoading(true);
                        try {
                            const response = await fetch(`${API_BASE_URL}/items/?item_id=${item.getId()}`, {
                                method: 'DELETE'
                            });
                            if (response.ok) {
                                if (onRefresh) onRefresh();
                                navigation.goBack();
                            } else {
                                throw new Error("Erreur suppression");
                            }
                        } catch (e) {
                            setNotification({ msg: "Impossible de supprimer", type: 'error' });
                            setLoading(false);
                        }
                    }
                }
            ]
        );
    };

    const handleAddToCart = () => {
        CartService.addItemToCart(item);
        setNotification({ msg: "Ajouté au panier", type: 'success' });
    };

    const inputStyle = { backgroundColor: theme.colors.background };

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <ScreenTitle title="Détail Produit" />

            <ScrollView contentContainerStyle={styles.scrollContent}>

                <View style={styles.headerInfo}>
                    <View style={styles.imageContainer}>
                        <Image
                            source={{ uri: item.getImageUrl() || undefined }}
                            style={[styles.image, { backgroundColor: theme.colors.surfaceVariant }]}
                            resizeMode="contain"
                        />
                    </View>

                    <View style={styles.mainInfo}>
                        <Text variant="headlineSmall" style={{ color: theme.colors.onBackground, fontWeight: 'bold' }}>
                            {item.getProductName()}
                        </Text>
                        {item.getBrands() ? (
                            <Text variant="bodyMedium" style={{ color: theme.colors.primary, marginTop: 4 }}>
                                {item.getBrands()}
                            </Text>
                        ) : null}
                        <Text variant="bodySmall" style={{ color: theme.colors.onSurfaceVariant, marginTop: 8 }}>
                            EAN: {item.getBarcodeId()}
                        </Text>
                    </View>
                </View>

                <Divider style={{ marginVertical: 20 }} />

                <Text variant="titleMedium" style={{ color: theme.colors.onBackground, marginBottom: 15, fontWeight: 'bold' }}>
                    Gérer votre produit
                </Text>

                <TextInput
                    label="Prix (€)"
                    value={formData.price}
                    onChangeText={t => setFormData({ ...formData, price: t })}
                    keyboardType="numeric"
                    style={[styles.input, inputStyle]}
                    mode="outlined"
                    textColor={theme.colors.onSurface}
                />

                <AppPicker
                    label="Rayon"
                    selectedValue={formData.rayon}
                    onValueChange={v => setFormData({ ...formData, rayon: v! })}
                    items={optRayons}
                />

                <AppPicker
                    label="Attribut"
                    selectedValue={formData.attribut}
                    onValueChange={v => setFormData({ ...formData, attribut: v! })}
                    items={optAttributs}
                />

                <AppPicker
                    label="Conservation"
                    selectedValue={formData.conservation}
                    onValueChange={v => setFormData({ ...formData, conservation: v! })}
                    items={optConservations}
                />

                <Divider style={{ marginVertical: 20 }} />

                {item.getIngredeintsText() ? (
                    <View style={{ marginBottom: 20 }}>
                        <Text variant="titleMedium" style={{ color: theme.colors.onBackground, marginBottom: 8, fontWeight: 'bold' }}>
                            Ingrédients
                        </Text>
                        <Surface style={[styles.ingredientsBox, { backgroundColor: theme.colors.elevation.level1 }]} elevation={1}>
                            <Text variant="bodyMedium" style={{ color: theme.colors.onSurfaceVariant }}>
                                {item.getIngredeintsText()}
                            </Text>
                        </Surface>
                    </View>
                ) : null}

                {/* Actions */}
                <View style={styles.actions}>
                    <Button
                        mode="contained"
                        onPress={handleUpdate}
                        loading={loading}
                        style={styles.btn}
                        icon="content-save"
                        buttonColor={theme.colors.primary}
                        textColor={theme.colors.onPrimary}
                    >
                        Enregistrer les modifications
                    </Button>

                    <Button
                        mode="contained-tonal"
                        onPress={handleAddToCart}
                        style={styles.btn}
                        icon="cart-plus"
                    >
                        Ajouter au panier
                    </Button>

                    <Button
                        mode="outlined"
                        onPress={handleDelete}
                        style={[styles.btn, { borderColor: theme.colors.error }]}
                        textColor={theme.colors.error}
                        icon="delete"
                    >
                        Supprimer de la base
                    </Button>
                </View>

            </ScrollView>
            <ErrorNotification
                message={notification?.msg || null}
                type={notification?.type}
                onClose={() => setNotification(null)}
            />
        </View>
    );
}

const styles = StyleSheet.create({
    container: { flex: 1 },
    scrollContent: { padding: 20, paddingBottom: 50 },

    headerInfo: {
        alignItems: 'center',
    },
    imageContainer: {
        marginBottom: 15,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.1,
        shadowRadius: 4,
        elevation: 3,
    },
    image: {
        width: 180,
        height: 180,
        borderRadius: 12
    },
    mainInfo: {
        alignItems: 'center',
        paddingHorizontal: 10,
    },

    input: {
        marginBottom: 12
    },

    ingredientsBox: {
        padding: 15,
        borderRadius: 8,
    },

    actions: {
        marginTop: 10,
        gap: 12
    },
    btn: {
        borderRadius: 8
    }
});