import { CameraView, useCameraPermissions, BarcodeScanningResult } from 'expo-camera';
import React, { useState, useEffect } from 'react';
import { Alert, Button, StyleSheet, Text, View, Modal, TextInput, Image, ActivityIndicator, ScrollView, TouchableOpacity } from 'react-native';
import { useTheme } from 'react-native-paper';
import { Item } from "../types/Item";
import { API_BASE_URL } from '../config';
import AppPicker from '../components/AppPicker';
import ScannedItem from '../interfaces/ScannedItem';
import ManualItemInput from '../components/ManualItemInput';
import ErrorNotification from '../components/ErrorNotification';
import { CartService } from '../services/CartService';

/*
Un écran pour scanner des articles via la caméra ou saisir manuellement un code-barres.
Fonctionnalités:
- Gère les permissions de la caméra.
- Permet de scanner des codes-barres et de récupérer les détails des articles depuis l'API.
- Permet la saisie manuelle des codes-barres.
- Gère les articles nouveaux nécessitant des détails supplémentaires avant ajout au panier.
- Affiche des notifications d'erreur ou de succès.
*/

export const showAddToCartConfirmation = (
    finalItem: Item,
    showNotification: (msg: string, type: 'error' | 'success') => void,
    onCartUpdate: () => void = () => { }
) => {
    const isAlreadyInCart = CartService.getCartItems().some(
        cartItem => cartItem.getItem().getBarcodeId() === finalItem.getBarcodeId()
    );

    const confirmationMessage = isAlreadyInCart
        ? `L'article "${finalItem.getProductName()}" est déjà dans le panier. Voulez-vous incrémenter sa quantité de 1 ?`
        : `Voulez-vous ajouter "${finalItem.getProductName()}" au panier ?`;

    Alert.alert(
        'Ajouter au Panier',
        confirmationMessage,
        [
            {
                text: 'Non',
                style: 'cancel',
                onPress: () => { onCartUpdate(); },
            },
            {
                text: isAlreadyInCart ? 'Incrémenter' : 'Ajouter',
                onPress: () => {
                    const updatedCartItem = CartService.addItemToCart(finalItem);
                    const message = isAlreadyInCart
                        ? `Quantité de "${finalItem.getProductName()}" incrémentée. Nouvelle quantité: ${updatedCartItem.getQuantity()}`
                        : `"${finalItem.getProductName()}" ajouté au panier.`;
                    showNotification(message, 'success');
                    onCartUpdate();
                },
            },
        ],
        { cancelable: false }
    );
};

export default function ScannerScreen() {
    const [permission, requestPermission] = useCameraPermissions();
    const [scanned, setScanned] = useState(false);
    const [isManualInputVisible, setIsManualInputVisible] = useState(false);
    const [isLoading, setIsLoading] = useState(false);
    const [notification, setNotification] = useState<{ msg: string, type: 'error' | 'success' } | null>(null);
    const [item, setItem] = useState<ScannedItem | null>(null);

    const [isModalVisible, setIsModalVisible] = useState(false);
    const [isScanningActive, setIsScanningActive] = useState(false);
    const [isNewItem, setIsNewItem] = useState(false);

    const [allRayons, setAllRayons] = useState<string[]>([]);
    const [allAttributs, setAllAttributs] = useState<string[]>([]);
    const [allConservations, setAllConservations] = useState<string[]>([]);

    const [newPrice, setNewPrice] = useState('');
    const [selectedRayon, setSelectedRayon] = useState<string | undefined>();
    const [selectedAttribut, setSelectedAttribut] = useState<string | undefined>();
    const [selectedConservation, setSelectedConservation] = useState<string | undefined>();

    const theme = useTheme(); // Récupération du thème actuel

    const showNotification = (msg: string, type: 'error' | 'success' = 'error') => {
        setNotification({ msg, type });
    };

    const handleManualSubmit = (barcode: string) => {
        setIsManualInputVisible(false);
        const mockResult = { data: barcode } as BarcodeScanningResult;
        handleBarcodeScanned(mockResult);
    };

    const fetchOptions = async (endpoint: string, setter: React.Dispatch<React.SetStateAction<string[]>>) => {
        try {
            const response = await fetch(`${API_BASE_URL}/items/options/${endpoint}`);
            if (!response.ok) throw new Error(`Erreur réseau pour ${endpoint}`);
            const data: string[] = await response.json();
            setter(data);
        } catch (error) {
            showNotification(`Erreur fetchOptions (${endpoint}): ${error}`, 'error')
        }
    };

    useEffect(() => {
        fetchOptions('rayons', setAllRayons);
        fetchOptions('attributs', setAllAttributs);
        fetchOptions('conservations', setAllConservations);
    }, []);

    if (!permission) {
        return (
            <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
                <ActivityIndicator size="large" color={theme.colors.primary} />
            </View>
        );
    }

    if (!permission.granted) {
        return (
            <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
                <Text style={[styles.permissionText, { color: theme.colors.onBackground }]}>Permission caméra nécessaire pour le scan</Text>
                <Button onPress={requestPermission} title="Accorder la permission" color={theme.colors.primary} />

                <View style={styles.manualContainerBlocked}>
                    <Text style={[styles.separatorText, { color: theme.colors.onBackground }]}>--- OU ---</Text>
                    <ManualItemInput onSubmit={handleManualSubmit} visible={true} />
                </View>
            </View>
        );
    }

    const handleBarcodeScanned = async (scanningResult: BarcodeScanningResult) => {
        if (scanned) return;
        setIsScanningActive(false);
        setScanned(true);
        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/${scanningResult.data}`);
            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Produit non trouvé");
            }

            const fetchedItem: ScannedItem = await response.json();
            const isNew = fetchedItem.rayon === null || (fetchedItem.price as number) === 0;
            setIsNewItem(isNew);
            setItem(fetchedItem);

            if (isNew) {
                setNewPrice(fetchedItem.price ? fetchedItem.price.toString() : '');
                setSelectedRayon(fetchedItem.rayon ? fetchedItem.rayon.toString() : '');
                setSelectedAttribut(fetchedItem.attribut ? fetchedItem.attribut.toString() : '');
                setSelectedConservation(fetchedItem.conservation ? fetchedItem.conservation.toString() : '');
                setIsModalVisible(true);
            } else {
                const existingItemObject = new Item(
                    fetchedItem.item_id as number, fetchedItem.barcode_id as string, fetchedItem.product_name as string,
                    fetchedItem.image_url as string, fetchedItem.small_image_url as string, fetchedItem.brands as string,
                    fetchedItem.nutriscore_grade as string, fetchedItem.product_quantity as number, fetchedItem.product_quantity_unit as string,
                    fetchedItem.ingredients_text as string, fetchedItem.allergens_imported as string, fetchedItem.price as number,
                    fetchedItem.rayon as string, fetchedItem.attribut as string, fetchedItem.conservation as string
                );
                showAddToCartConfirmation(existingItemObject, showNotification, () => { setScanned(false); });
            }
        } catch (error) {
            showNotification(`Impossible de récupérer les détails pour ${scanningResult.data}. ${error}`, 'error');
            setScanned(false);
        } finally {
            setIsLoading(false);
        }
    };

    const handleCloseModal = () => {
        setIsModalVisible(false);
        setItem(null);
        setScanned(false);
        setNewPrice('');
        setSelectedRayon(undefined);
        setSelectedAttribut(undefined);
        setSelectedConservation(undefined);
    };

    const handleSubmitUpdates = async () => {
        if (!item) return;
        if (!isNewItem) {
            handleCloseModal();
            return;
        }

        const priceValue = parseFloat(newPrice.replace(',', '.'));
        if (isNaN(priceValue) || priceValue <= 0) {
            showNotification("Le prix doit être un nombre valide > 0.", 'error');
            return;
        }
        if (!selectedRayon || !selectedAttribut || !selectedConservation) {
            showNotification("Tous les champs sont obligatoires.", 'error');
            return;
        }

        const updateDetails = {
            price: priceValue,
            rayon: selectedRayon,
            attribut: selectedAttribut,
            conservation: selectedConservation,
        };

        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/details/${item.barcode_id}`, {
                method: 'PATCH',
                headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                body: JSON.stringify(updateDetails),
            });

            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Erreur lors de la mise à jour");
            }

            const updatedItemObject = new Item(
                item.item_id as number, item.barcode_id as string, item.product_name as string,
                item.image_url as string, item.small_image_url as string, item.brands as string,
                item.nutriscore_grade as string, item.product_quantity as number, item.product_quantity_unit as string,
                item.ingredients_text as string, item.allergens_imported as string, priceValue,
                selectedRayon as string, selectedAttribut as string, selectedConservation as string
            );

            handleCloseModal();
            setTimeout(() => {
                showAddToCartConfirmation(updatedItemObject, showNotification, () => { setScanned(false); });
            }, 500);

        } catch (error) {
            showNotification(`Échec de la mise à jour: ${error}`, 'error')
        } finally {
            setIsLoading(false);
        }
    };

    const renderStartButton = () => {
        if (!isScanningActive && !isModalVisible && !isLoading && !isManualInputVisible) {
            return (
                <View style={styles.buttonsWrapper}>
                    <TouchableOpacity
                        style={[styles.actionButton, { backgroundColor: theme.colors.primary }]}
                        onPress={() => setIsScanningActive(true)}
                    >
                        <Text style={[styles.actionButtonText, { color: theme.colors.onPrimary }]}>Scanner</Text>
                    </TouchableOpacity>

                    <TouchableOpacity
                        style={[styles.actionButton, { backgroundColor: theme.colors.surfaceVariant }]}
                        onPress={() => setIsManualInputVisible(true)}
                    >
                        <Text style={[styles.actionButtonText, { color: theme.colors.onSurfaceVariant }]}>Saisie Manuelle</Text>
                    </TouchableOpacity>
                </View>
            );
        }
        return null;
    };

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <CameraView
                style={styles.camera}
                facing="back"
                onBarcodeScanned={isScanningActive && !scanned ? handleBarcodeScanned : undefined}
                barcodeScannerSettings={{ barcodeTypes: ['ean13', 'qr'] }}
            />

            {isLoading && !isModalVisible &&
                <ActivityIndicator size="large" color={theme.colors.primary} style={styles.loading} />
            }

            {renderStartButton()}

            <Modal
                visible={isManualInputVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={() => setIsManualInputVisible(false)}
            >
                <View style={styles.modalContainer}>
                    <ManualItemInput
                        onSubmit={handleManualSubmit}
                        onCancel={() => setIsManualInputVisible(false)}
                        visible={true}
                    />
                </View>
            </Modal>

            <Modal
                visible={isModalVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={handleCloseModal}
            >
                <View style={styles.modalContainer}>
                    <View style={[styles.modalView, { backgroundColor: theme.colors.surface }]}>
                        <ScrollView contentContainerStyle={styles.modalScrollContent}>
                            {item && (
                                <>
                                    <Text style={[styles.modalTitle, { color: theme.colors.onSurface }]}>
                                        Nouvel Article
                                    </Text>

                                    <Image
                                        style={[styles.itemImage, { backgroundColor: theme.colors.surfaceVariant }]}
                                        source={{ uri: item.small_image_url || undefined }}
                                        resizeMode="contain"
                                    />

                                    <Text style={[styles.itemTextName, { color: theme.colors.onSurface }]}>{item.product_name}</Text>

                                    <Text style={[styles.label, { color: theme.colors.onSurface }]}>Prix (€)</Text>
                                    <TextInput
                                        style={[styles.input, {
                                            backgroundColor: theme.colors.elevation.level1,
                                            color: theme.colors.onSurface,
                                            borderColor: theme.colors.outline
                                        }]}
                                        keyboardType="numeric"
                                        value={newPrice}
                                        onChangeText={setNewPrice}
                                        placeholder="0.00"
                                        placeholderTextColor={theme.colors.onSurfaceDisabled}
                                    />

                                    <AppPicker
                                        label="Rayon"
                                        selectedValue={selectedRayon}
                                        onValueChange={setSelectedRayon}
                                        items={allRayons}
                                    />

                                    <AppPicker
                                        label="Attribut"
                                        selectedValue={selectedAttribut}
                                        onValueChange={setSelectedAttribut}
                                        items={allAttributs}
                                    />

                                    <AppPicker
                                        label="Conservation"
                                        selectedValue={selectedConservation}
                                        onValueChange={setSelectedConservation}
                                        items={allConservations}
                                    />

                                    <View style={styles.buttonContainer}>
                                        <Button title="Annuler" onPress={handleCloseModal} color={theme.colors.error} />
                                        {isLoading && isModalVisible ? (
                                            <ActivityIndicator color={theme.colors.primary} style={{ paddingHorizontal: 20 }} />
                                        ) : (
                                            <Button title="Valider" onPress={handleSubmitUpdates} color={theme.colors.primary} />
                                        )}
                                    </View>
                                </>
                            )}
                        </ScrollView>
                    </View>
                </View>
            </Modal>
            <ErrorNotification
                message={notification?.msg || null}
                type={notification?.type}
                onClose={() => setNotification(null)}
            />
        </View>
    );
}

const styles = StyleSheet.create({
    container: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
    },
    permissionText: {
        textAlign: 'center',
        margin: 20,
        fontSize: 16,
    },
    camera: {
        ...StyleSheet.absoluteFillObject,
    },
    loading: {
        position: 'absolute',
    },
    buttonsWrapper: {
        position: 'absolute',
        bottom: 50,
        width: '100%',
        flexDirection: 'row',
        justifyContent: 'space-between',
        paddingHorizontal: 20,
        gap: 15,
    },
    actionButton: {
        flex: 1,
        paddingVertical: 18,
        borderRadius: 12,
        alignItems: 'center',
        justifyContent: 'center',
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 4 },
        shadowOpacity: 0.3,
        shadowRadius: 4,
        elevation: 6,
    },
    actionButtonText: {
        fontSize: 16,
        fontWeight: 'bold',
        letterSpacing: 0.5,
    },
    manualContainerBlocked: {
        width: '100%',
        marginTop: 30,
        alignItems: 'center',
    },
    separatorText: {
        marginVertical: 10,
        fontWeight: 'bold',
    },
    modalContainer: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: 'rgba(0,0,0,0.6)',
    },
    modalView: {
        width: '90%',
        maxHeight: '85%',
        borderRadius: 20,
        padding: 20,
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.25,
        shadowRadius: 4,
        elevation: 5,
    },
    modalScrollContent: {
        alignItems: 'center',
    },
    modalTitle: {
        fontSize: 22,
        fontWeight: 'bold',
        marginBottom: 15,
    },
    itemImage: {
        width: 150,
        height: 150,
        marginBottom: 15,
        borderRadius: 8,
    },
    itemTextName: {
        fontSize: 18,
        fontWeight: '600',
        textAlign: 'center',
        marginBottom: 20,
    },
    label: {
        fontWeight: 'bold',
        marginTop: 10,
        width: '100%'
    },
    input: {
        width: '100%',
        height: 50,
        borderWidth: 1,
        borderRadius: 8,
        paddingHorizontal: 15,
        marginBottom: 10,
        marginTop: 5,
        fontSize: 16,
    },
    buttonContainer: {
        flexDirection: 'row',
        justifyContent: 'space-around',
        width: '100%',
        marginTop: 25,
        marginBottom: 10,
    }
});