import React from 'react';
import { View, StyleSheet, ScrollView } from 'react-native';
import { Text, useTheme } from 'react-native-paper';
import { useRoute } from '@react-navigation/native';
import ScreenTitle from '../components/ScreenTitle';
import { OrderDetailRouteProp } from '../types/OrderDetailRouteProp';

/*
Un écran pour afficher les détails d'une commande spécifique.
Fonctionnalités:
- Affiche les informations détaillées de la commande, y compris les articles achetés, les quantités et les prix.
- Calcule et affiche le total HT, la TVA et le total TTC.
*/

export default function OrderDetailScreen() {
    const route = useRoute<OrderDetailRouteProp>();
    const theme = useTheme();
    const { payment } = route.params;

    const formatDate = (dateString: string | null | undefined) => {
        if (!dateString) return '';
        return new Date(dateString).toLocaleDateString('fr-FR', {
            day: 'numeric', month: 'long', year: 'numeric', hour: '2-digit', minute: '2-digit'
        });
    };

    const formatPrice = (price: number) => {
        return new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR' }).format(price);
    };

    const totalHT = payment.total_price;
    const tvaAmount = totalHT * 0.20;
    const totalTTC = totalHT + tvaAmount;

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>

            <ScreenTitle
                title={`Commande #${payment.id.slice(-6).toUpperCase()}`}
                subtitle={formatDate(payment.checkout_date)}
            />

            <ScrollView style={styles.scrollContent}>
                <View style={[styles.invoiceCard, { backgroundColor: theme.colors.elevation.level1 }]}>
                    <View style={styles.rowHeader}>
                        <Text style={[styles.colProduct, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Produit</Text>
                        <Text style={[styles.colQty, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Qté</Text>
                        <Text style={[styles.colTotal, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Prix</Text>
                    </View>
                    <View style={[styles.divider, { backgroundColor: theme.colors.outlineVariant }]} />

                    {payment.purchased_items.map((pItem, index) => (
                        <View key={index} style={styles.rowItem}>
                            <Text style={[styles.colProduct, { color: theme.colors.onSurface }]} numberOfLines={1}>
                                {pItem.item.product_name}
                            </Text>
                            <Text style={[styles.colQty, { color: theme.colors.onSurfaceVariant }]}>x{pItem.amount}</Text>
                            <Text style={[styles.colTotal, { color: theme.colors.onSurface }]}>
                                {formatPrice(pItem.amount * pItem.item.price)}
                            </Text>
                        </View>
                    ))}

                    <View style={[styles.divider, { marginVertical: 15, backgroundColor: theme.colors.outlineVariant }]} />

                    <View style={styles.summaryRow}>
                        <Text style={[styles.summaryLabel, { color: theme.colors.onSurfaceVariant }]}>Total HT</Text>
                        <Text style={[styles.summaryValue, { color: theme.colors.onSurface }]}>{formatPrice(totalHT)}</Text>
                    </View>
                    <View style={styles.summaryRow}>
                        <Text style={[styles.summaryLabel, { color: theme.colors.onSurfaceVariant }]}>TVA (20%)</Text>
                        <Text style={[styles.summaryValue, { color: theme.colors.onSurface }]}>{formatPrice(tvaAmount)}</Text>
                    </View>

                    <View style={[styles.totalRow, { borderColor: theme.colors.outlineVariant }]}>
                        <Text style={[styles.totalLabel, { color: theme.colors.onSurface }]}>Total TTC</Text>
                        <Text style={[styles.totalValue, { color: theme.colors.primary }]}>{formatPrice(totalTTC)}</Text>
                    </View>
                </View>
            </ScrollView>
        </View>
    );
}

const styles = StyleSheet.create({
    container: {
        flex: 1
    },

    scrollContent: {
        padding: 20
    },

    invoiceCard: {
        borderRadius: 16,
        padding: 20,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.05,
        shadowRadius: 8,
        elevation: 2,
        marginBottom: 50
    },

    rowHeader: {
        flexDirection: 'row',
        marginBottom: 10
    },

    headerText: {
        fontSize: 12,
        fontWeight: '600',
        textTransform: 'uppercase'
    },

    rowItem: {
        flexDirection: 'row',
        paddingVertical: 12
    },

    colProduct: {
        flex: 1,
        fontSize: 15
    },

    colQty: {
        width: 40,
        textAlign: 'center',
        fontSize: 15
    },
    colTotal: {
        width: 70,
        textAlign: 'right',
        fontSize: 15,
        fontWeight: '600'
    },
    divider: {
        height: 1
    },
    summaryRow: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginBottom: 8
    },
    summaryLabel: {
        fontSize: 15
    },
    summaryValue: {
        fontSize: 15
    },
    totalRow: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginTop: 10,
        paddingTop: 10,
        borderTopWidth: 1
    },
    totalLabel: {
        fontSize: 18,
        fontWeight: '700'
    },
    totalValue: {
        fontSize: 22,
        fontWeight: '800'
    },
});