import React, { useState, useEffect, useCallback } from 'react';
import { View, Text, ScrollView, StyleSheet, Image, TouchableOpacity, ActivityIndicator } from 'react-native';
import { FontAwesome, MaterialCommunityIcons } from '@expo/vector-icons';
import { CartItem } from '../types/CartItem';
import { CartService } from '../services/CartService';
import { useNavigation } from '@react-navigation/native';
import { StackNavigationProp } from '@react-navigation/stack';
import { useTheme } from 'react-native-paper';

type CartStackParamList = {
    CartMain: undefined;
    RecapScreen: undefined;
};

interface CartItemRowProps {
    cartItem: CartItem;
    onQuantityChange: () => void;
    theme: any;
}

interface CartScreenProps {
    onValidateCart?: () => void;
}

const CartItemRow: React.FC<CartItemRowProps> = ({ cartItem, onQuantityChange, theme }) => {
    const item = cartItem.getItem();
    const quantity = cartItem.getQuantity();
    const totalPrice = cartItem.getTotalPrice();

    const handleDecrement = () => {
        CartService.decrementItemQuantity(item.getBarcodeId());
        onQuantityChange();
    };
    const handleIncrement = () => {
        CartService.incrementItemQuantity(item.getBarcodeId());
        onQuantityChange();
    };
    const handleRemove = () => {
        CartService.removeItemFromCart(item.getBarcodeId());
        onQuantityChange();
    };

    return (
        <View style={[styles.card, { backgroundColor: theme.colors.elevation.level1 }]}>
            <Image source={{ uri: item.getSmallImageUrl() as string }} style={styles.itemImage} resizeMode="contain" />
            <View style={styles.cardContent}>
                <View style={styles.cardHeader}>
                    <Text style={[styles.itemName, { color: theme.colors.onSurface }]} numberOfLines={2}>{item.getProductName()}</Text>
                    <TouchableOpacity onPress={handleRemove} style={styles.deleteButton}>
                        <MaterialCommunityIcons name="trash-can-outline" size={22} color={theme.colors.error} />
                    </TouchableOpacity>
                </View>

                <Text style={[styles.unitPrice, { color: theme.colors.onSurfaceVariant }]}>PU: {(item.getPrice() as number).toFixed(2)} €</Text>

                <View style={styles.cardFooter}>
                    <View style={[styles.quantityContainer, { backgroundColor: theme.colors.elevation.level3 }]}>
                        <TouchableOpacity onPress={handleDecrement} style={styles.qtyBtn}>
                            <FontAwesome name="minus" size={12} color={theme.colors.onSurface} />
                        </TouchableOpacity>
                        <Text style={[styles.qtyText, { color: theme.colors.onSurface }]}>{quantity}</Text>
                        <TouchableOpacity onPress={handleIncrement} style={styles.qtyBtn}>
                            <FontAwesome name="plus" size={12} color={theme.colors.onSurface} />
                        </TouchableOpacity>
                    </View>
                    <Text style={[styles.rowTotal, { color: theme.colors.primary }]}>{(totalPrice as number).toFixed(2)} €</Text>
                </View>
            </View>
        </View>
    );
};

const CartScreen: React.FC<CartScreenProps> = ({ onValidateCart }) => {
    const [cartItems, setCartItems] = useState<CartItem[]>([]);
    const [isLoading, setIsLoading] = useState<boolean>(true);
    const navigation = useNavigation<StackNavigationProp<CartStackParamList>>();
    const theme = useTheme();

    const handleValidateCart = () => { navigation.navigate('RecapScreen'); };
    const refreshCart = useCallback(() => { setCartItems(CartService.getCartItems()); }, []);
    const formatPrice = (price: number): string => { return new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR', }).format(price); };

    useEffect(() => {
        const initCart = async () => {
            setIsLoading(true);
            try { await CartService.fetchCartFromServer(); refreshCart(); }
            catch (error) { console.error("Impossible de récupérer le panier serveur", error); }
            finally { setIsLoading(false); }
        };
        initCart();
    }, [refreshCart]);

    const grandTotal = CartService.getTotalPrice() as number;
    const distinctCount = CartService.getDistinctItemCount();

    if (isLoading) {
        return (
            <View style={[styles.container, styles.loadingContainer, { backgroundColor: theme.colors.background }]}>
                <ActivityIndicator size="large" color={theme.colors.primary} />
                <Text style={[styles.loadingText, { color: theme.colors.onSurfaceVariant }]}>Chargement du panier...</Text>
            </View>
        );
    }

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <View style={[styles.headerContainer, { backgroundColor: theme.colors.background, borderColor: theme.colors.outlineVariant }]}>
                <Text style={[styles.headerTitle, { color: theme.colors.onBackground }]}>Mon Panier</Text>
            </View>

            <ScrollView style={styles.scrollView} contentContainerStyle={{ paddingBottom: 100 }}>
                {cartItems.length === 0 ? (
                    <View style={styles.emptyContainer}>
                        <MaterialCommunityIcons name="cart-off" size={60} color={theme.colors.onSurfaceDisabled} />
                        <Text style={[styles.emptyText, { color: theme.colors.onSurfaceVariant }]}>Votre panier est vide</Text>
                    </View>
                ) : (
                    cartItems.map((cartItem) => (
                        <CartItemRow
                            key={cartItem.getItem().getBarcodeId() as string}
                            cartItem={cartItem}
                            onQuantityChange={refreshCart}
                            theme={theme}
                        />
                    ))
                )}
            </ScrollView>

            {distinctCount > 0 && (
                <View style={[styles.footer, { backgroundColor: theme.colors.surface, borderColor: theme.colors.outlineVariant }]}>
                    <Text style={[styles.qtyText, { color: theme.colors.onSurface }]}>Articles distincts: {distinctCount}</Text>
                    <Text style={[styles.rowTotal, { color: theme.colors.primary }]}>Total du panier: {formatPrice(grandTotal)}</Text>

                    <TouchableOpacity
                        style={[styles.validateButton, { backgroundColor: theme.colors.primary }]}
                        onPress={handleValidateCart}
                    >
                        <Text style={[styles.validateButtonText, { color: theme.colors.onPrimary }]}>Valider le panier</Text>
                        <FontAwesome name="arrow-right" size={16} color={theme.colors.onPrimary} style={{ marginLeft: 10 }} />
                    </TouchableOpacity>
                </View>
            )}
        </View>
    );
};

const styles = StyleSheet.create({
    container: {
        flex: 1
    },

    loadingContainer: {
        justifyContent: 'center',
        alignItems: 'center'
    },

    loadingText: {
        marginTop: 10,
        fontSize: 16
    },

    headerContainer: {
        padding: 20,
        paddingTop: 60,
        borderBottomWidth: 1
    },

    headerTitle: {
        fontSize: 28,
        fontWeight: '800'
    },

    scrollView: {
        flex: 1,
        padding: 15,
        marginBottom: 75
    },

    card: {
        flexDirection: 'row',
        borderRadius: 12,
        padding: 12,
        marginBottom: 12,
        shadowColor: "#000",
        shadowOffset: {
            width: 0,
            height: 2
        },
        shadowOpacity: 0.1,
        shadowRadius: 4,
        elevation: 3,
    },

    itemImage: {
        width: 70,
        height: 70,
        borderRadius: 8,
        backgroundColor: '#F2F2F7'
    },

    cardContent: {
        flex: 1,
        marginLeft: 12,
        justifyContent: 'space-between'
    },

    cardHeader: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'flex-start'
    },

    itemName: {
        fontSize: 16,
        fontWeight: '600',
        flex: 1,
        marginRight: 8
    },

    deleteButton: {
        padding: 4
    },

    unitPrice: {
        fontSize: 12,
        marginTop: 4
    },

    cardFooter: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        alignItems: 'center',
        marginTop: 8
    },

    quantityContainer: {
        flexDirection: 'row',
        alignItems: 'center',
        borderRadius: 8
    },

    qtyBtn: {
        padding: 8,
        alignItems: 'center',
        justifyContent: 'center',
        width: 32
    },

    qtyText: {
        fontSize: 14,
        fontWeight: '600',
        paddingHorizontal: 8
    },

    rowTotal: {
        fontSize: 16,
        fontWeight: '700'
    },

    emptyContainer: {
        alignItems: 'center',
        justifyContent: 'center',
        marginTop: 100
    },

    emptyText: {
        marginTop: 16,
        fontSize: 18
    },

    footer: {
        position: 'absolute',
        bottom: 0,
        left: 0,
        right: 0,
        padding: 20,
        paddingBottom: 30,
        borderTopWidth: 1,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: -3 },
        shadowOpacity: 0.1,
        shadowRadius: 3
    },
    validateButton: {
        marginTop: 15,
        paddingVertical: 12,
        paddingHorizontal: 20,
        borderRadius: 25,
        flexDirection: 'row',
        alignItems: 'center',
        justifyContent: 'center',
        elevation: 3,
    },

    validateButtonText: {
        fontSize: 18,
        fontWeight: 'bold'
    }
});

export default CartScreen;