import React, { useEffect, useState } from 'react';
import { View, Text, StyleSheet, ScrollView, ActivityIndicator, TouchableOpacity } from 'react-native';
import { useStripe } from "@stripe/stripe-react-native";
import { CartService } from '../services/CartService';
import { CartItem } from '../types/CartItem';
import { API_BASE_URL, customer_ID } from "../config";
import { useNavigation, useRoute } from '@react-navigation/native';
import ErrorNotification from '../components/ErrorNotification';
import { Payment } from '../services/PaiementService';
import { useTheme } from 'react-native-paper';

export default function RecapScreen() {
    const [cartItems, setCartItems] = useState<CartItem[]>([]);
    const [loading, setLoading] = useState(false);
    const { initPaymentSheet, presentPaymentSheet } = useStripe();
    const navigation = useNavigation();
    const route = useRoute();
    const [notification, setNotification] = useState<{ msg: string, type: 'error' | 'success' } | null>(null);
    const [paymentIntentId, setPaymentIntentId] = useState<string | null>(null);
    const userId = customer_ID;
    const theme = useTheme();

    const params = route.params as { payment?: Payment };
    const historyPayment = params?.payment;

    useEffect(() => {
        if (historyPayment) {
            const mappedItems = historyPayment.purchased_items.map(pItem => ({
                getItem: () => ({ getProductName: () => pItem.item.product_name, getId: () => 0, getPrice: () => pItem.item.price }),
                getQuantity: () => pItem.amount, getTotalPrice: () => pItem.amount * pItem.item.price
            }));
            setCartItems(mappedItems as unknown as CartItem[]);
        } else {
            const items = CartService.getCartItems();
            setCartItems(items);
            initializePaymentSheet(items);
        }
    }, [historyPayment]);

    const showNotification = (msg: string, type: 'error' | 'success' = 'error') => { setNotification({ msg, type }); };
    const formatDate = (dateString: string | null | undefined) => {
        if (!dateString) return '';
        return new Date(dateString).toLocaleDateString('fr-FR', { day: 'numeric', month: 'long', year: 'numeric', hour: '2-digit', minute: '2-digit' });
    };
    const totalHT = historyPayment ? historyPayment.total_price : CartService.getTotalPrice() as number;
    const tvaAmount = totalHT * 0.20;
    const totalTTC = totalHT + tvaAmount;

    const fetchPaymentSheetParams = async (itemsToPay: CartItem[]) => {
        const pendingItems = itemsToPay.map(ci => ({ id: ci.getItem().getId(), amount: ci.getQuantity() }));
        if (pendingItems.length === 0) return null;
        try {
            const response = await fetch(`${API_BASE_URL}/payments/`, {
                method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ "pending_items": pendingItems, "customer_id": userId })
            });
            if (!response.ok) { const errorText = await response.text(); console.error("Erreur backend paiement", errorText); showNotification(`Erreur paiement: ${errorText}`, 'error'); return null; }
            const { paymentIntent, ephemeralKey, customer } = await response.json();
            return { paymentIntent, ephemeralKey, customer };
        } catch (error) { showNotification("Erreur réseau lors de l'initialisation du paiement.", 'error'); return null; }
    };

    const initializePaymentSheet = async (itemsToPay: CartItem[]) => {
        if (historyPayment) return;
        setLoading(true);
        const params = await fetchPaymentSheetParams(itemsToPay);
        if (!params) { setLoading(false); return; }
        const { paymentIntent, ephemeralKey, customer } = params;
        setPaymentIntentId(paymentIntent);
        const { error } = await initPaymentSheet({
            merchantDisplayName: "Mon Supermarché", customerId: customer, customerEphemeralKeySecret: ephemeralKey, paymentIntentClientSecret: paymentIntent, allowsDelayedPaymentMethods: false, defaultBillingDetails: { name: 'Client Test', }
        });
        if (error) { console.log("Erreur init Stripe", error); showNotification("Erreur d'initialisation Stripe", 'error'); }
        setLoading(false);
    };

    const openPaymentSheet = async () => {
        const { error } = await presentPaymentSheet();
        if (error) { showNotification(`Paiement annulé ou échoué: ${error.message}`, 'error'); }
        else {
            if (!paymentIntentId) { showNotification("Erreur: ID de paiement introuvable", 'error'); return; }
            try {
                const cleanId = paymentIntentId.split('_secret_')[0];
                const response = await fetch(`${API_BASE_URL}/payments/check/${cleanId}`, { method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ "customer_id": userId }) });
                if (response.ok) { showNotification("Commande validée avec succès !", 'success'); CartService.clearCart(); setCartItems([]); navigation.navigate('HistoryScreen' as never); }
                else { showNotification("Erreur lors de la validation de la commande", 'error'); }
            } catch (err) { console.error("Erreur validation BDD", err); showNotification("Erreur réseau lors de la validation", 'error'); }
        }
    };

    const formatPrice = (price: number) => { return new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR' }).format(price); };

    return (
        <View style={[styles.container, { backgroundColor: theme.colors.background }]}>
            <ScrollView style={styles.scrollContent}>
                <View style={styles.headerSection}>
                    <Text style={[styles.title, { color: theme.colors.onBackground }]}>
                        {historyPayment ? `Commande #${historyPayment.id.slice(-6).toUpperCase()}` : "Facture"}
                    </Text>
                    {historyPayment && (
                        <Text style={[styles.dateSubtitle, { color: theme.colors.onSurfaceVariant }]}>
                            {formatDate(historyPayment.checkout_date)}
                        </Text>
                    )}
                </View>

                <View style={[styles.invoiceCard, { backgroundColor: theme.colors.elevation.level1 }]}>
                    <View style={styles.rowHeader}>
                        <Text style={[styles.colProduct, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Produit</Text>
                        <Text style={[styles.colQty, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Qté</Text>
                        <Text style={[styles.colTotal, styles.headerText, { color: theme.colors.onSurfaceVariant }]}>Prix</Text>
                    </View>
                    <View style={[styles.divider, { backgroundColor: theme.colors.outlineVariant }]} />

                    {cartItems.map((ci, index) => (
                        <View key={index} style={styles.rowItem}>
                            <Text style={[styles.colProduct, { color: theme.colors.onSurface }]} numberOfLines={1}>
                                {ci.getItem().getProductName()}
                            </Text>
                            <Text style={[styles.colQty, { color: theme.colors.onSurfaceVariant }]}>x{ci.getQuantity()}</Text>
                            <Text style={[styles.colTotal, { color: theme.colors.onSurface }]}>
                                {(ci.getTotalPrice() as number).toFixed(2)} €
                            </Text>
                        </View>
                    ))}

                    <View style={[styles.divider, { marginVertical: 15, backgroundColor: theme.colors.outlineVariant }]} />

                    <View style={styles.summaryRow}>
                        <Text style={[styles.summaryLabel, { color: theme.colors.onSurfaceVariant }]}>Total HT</Text>
                        <Text style={[styles.summaryValue, { color: theme.colors.onSurface }]}>{formatPrice(totalHT)}</Text>
                    </View>
                    <View style={styles.summaryRow}>
                        <Text style={[styles.summaryLabel, { color: theme.colors.onSurfaceVariant }]}>TVA (20%)</Text>
                        <Text style={[styles.summaryValue, { color: theme.colors.onSurface }]}>{formatPrice(tvaAmount)}</Text>
                    </View>

                    <View style={[styles.totalRow, { borderColor: theme.colors.outlineVariant }]}>
                        <Text style={[styles.totalLabel, { color: theme.colors.onSurface }]}>Total TTC</Text>
                        <Text style={[styles.totalValue, { color: theme.colors.primary }]}>{formatPrice(totalTTC)}</Text>
                    </View>
                </View>
            </ScrollView>

            {!historyPayment && (
                <View style={[styles.footer, { backgroundColor: theme.colors.surface, borderColor: theme.colors.outlineVariant }]}>
                    <TouchableOpacity
                        style={[styles.payButton, { backgroundColor: '#34C759' }, (loading || cartItems.length === 0) && styles.disabledButton]}
                        onPress={openPaymentSheet}
                        disabled={loading || cartItems.length === 0}
                    >
                        {loading ? <ActivityIndicator color="#fff" /> : <Text style={styles.payButtonText}>Payer {formatPrice(totalTTC)}</Text>}
                    </TouchableOpacity>
                </View>
            )}
            <ErrorNotification message={notification?.msg || null} type={notification?.type} onClose={() => setNotification(null)} />
        </View>
    );
}

const styles = StyleSheet.create({
    container: { flex: 1 },
    scrollContent: { padding: 20 },
    headerSection: { marginBottom: 20 },
    title: { fontSize: 28, fontWeight: '800' },
    dateSubtitle: { fontSize: 16, marginTop: 4, fontWeight: '500' },
    invoiceCard: { borderRadius: 16, padding: 20, shadowColor: "#000", shadowOffset: { width: 0, height: 2 }, shadowOpacity: 0.05, shadowRadius: 8, elevation: 2, marginBottom: 50 },
    rowHeader: { flexDirection: 'row', marginBottom: 10 },
    headerText: { fontSize: 12, fontWeight: '600', textTransform: 'uppercase' },
    rowItem: { flexDirection: 'row', paddingVertical: 12 },
    colProduct: { flex: 1, fontSize: 15 },
    colQty: { width: 40, textAlign: 'center', fontSize: 15 },
    colTotal: { width: 70, textAlign: 'right', fontSize: 15, fontWeight: '600' },
    divider: { height: 1 },
    summaryRow: { flexDirection: 'row', justifyContent: 'space-between', marginBottom: 8 },
    summaryLabel: { fontSize: 15 },
    summaryValue: { fontSize: 15 },
    totalRow: { flexDirection: 'row', justifyContent: 'space-between', marginTop: 10, paddingTop: 10, borderTopWidth: 1 },
    totalLabel: { fontSize: 18, fontWeight: '700' },
    totalValue: { fontSize: 22, fontWeight: '800' },
    footer: { padding: 20, borderTopWidth: 1 },
    payButton: { borderRadius: 14, paddingVertical: 16, alignItems: 'center' },
    disabledButton: { opacity: 0.6 },
    payButtonText: { color: '#fff', fontSize: 18, fontWeight: '700' }
});