from sqlalchemy.orm import Session, joinedload

from ..database import engine
from ..exceptions import NotFoundException
from ..items.models import Item
from .models import Cart, CartItem
from .schemas import CartItemCreate

def get_or_create_cart(db: Session) -> Cart:
    """
    Récupère le panier unique ou le créer s'il n'existe pas.
    """
    cart = db.query(Cart).first()

    if not cart:
        cart = Cart()
        db.add(cart)
        db.commit()
        db.refresh(cart)
        
    return cart

def add_item_to_cart(item_in: CartItemCreate) -> Cart:
    """
    Ajoute un item (ou met à jour sa quantité) au panier unique.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db)
        item_db = db.query(Item).filter(Item.item_id == item_in.item_id).first()
        if not item_db:
            raise NotFoundException(detail="Item not found")

        cart_item = db.query(CartItem).filter(
            CartItem.cart_id == cart.id,
            CartItem.item_id == item_in.item_id
        ).first()

        if cart_item:
            cart_item.quantity += item_in.quantity
        else:
            cart_item = CartItem(
                cart_id=cart.id,
                item_id=item_in.item_id,
                quantity=item_in.quantity
            )
            db.add(cart_item)

        db.commit()
        return get_cart()

def remove_item_from_cart(item_id: int) -> Cart:
    """
    Supprime un item du panier unique.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db)

        cart_item = db.query(CartItem).filter(
            CartItem.cart_id == cart.id,
            CartItem.item_id == item_id
        ).first()

        if not cart_item:
            raise NotFoundException(detail="Item not found in cart")

        db.delete(cart_item)
        db.commit()
        
        return get_cart()

def get_cart() -> Cart:
    """
    Récupère le contenu complet du panier unique.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db)
        
        cart_id = cart.id
        db.expire(cart) 
        
        cart_full = db.query(Cart).options(
            joinedload(Cart.items).joinedload(CartItem.item)
        ).filter(Cart.id == cart_id).first()

        return cart_full

def clear_cart() -> Cart:
    """
    Vide entièrement le panier unique.
    """
    with Session(engine) as db:
        cart = get_or_create_cart(db)
        
        db.query(CartItem).filter(CartItem.cart_id == cart.id).delete()
        db.commit()
        
        db.refresh(cart)
        return cart