import { CameraView, useCameraPermissions, BarcodeScanningResult } from 'expo-camera';
import React, { useState, useEffect } from 'react';
import { Alert, Button, StyleSheet, Text, View, Modal, TextInput, Image, ActivityIndicator, ScrollView, TouchableOpacity } from 'react-native';
import { Item } from "../types/Item";
import { API_BASE_URL } from '../config';
import AppPicker from '../components/AppPicker';
import ScannedItem from '../interfaces/ScannedItem';
import ManualItemInput from '../components/ManualItemInput';
import ErrorNotification from '../components/ErrorNotification';
import { CartService } from '../services/CartService';

export const showAddToCartConfirmation = (
    finalItem: Item,
    showNotification: (msg: string, type: 'error' | 'success') => void,
    onCartUpdate: () => void = () => { }
) => {
    const isAlreadyInCart = CartService.getCartItems().some(
        cartItem => cartItem.getItem().getBarcodeId() === finalItem.getBarcodeId()
    );

    const confirmationMessage = isAlreadyInCart
        ? `L'article "${finalItem.getProductName()}" est déjà dans le panier. Voulez-vous incrémenter sa quantité de 1 ?`
        : `Voulez-vous ajouter "${finalItem.getProductName()}" au panier ?`;

    Alert.alert(
        'Ajouter au Panier',
        confirmationMessage,
        [
            {
                text: 'Non',
                style: 'cancel',
                onPress: () => {
                    console.log('Ajout au panier annulé.');
                    onCartUpdate();
                },
            },
            {
                text: isAlreadyInCart ? 'Incrémenter' : 'Ajouter',
                onPress: () => {
                    const updatedCartItem = CartService.addItemToCart(finalItem);

                    const message = isAlreadyInCart
                        ? `Quantité de "${finalItem.getProductName()}" incrémentée. Nouvelle quantité: ${updatedCartItem.getQuantity()}`
                        : `"${finalItem.getProductName()}" ajouté au panier.`;

                    console.log(message);
                    showNotification(message, 'success');
                    onCartUpdate();
                },
            },
        ],
        { cancelable: false }
    );
};

export default function ScannerScreen() {
    const [permission, requestPermission] = useCameraPermissions();
    const [scanned, setScanned] = useState(false);
    const [isManualInputVisible, setIsManualInputVisible] = useState(false); const [isLoading, setIsLoading] = useState(false);
    const [notification, setNotification] = useState<{ msg: string, type: 'error' | 'success' } | null>(null);
    const [item, setItem] = useState<ScannedItem | null>(null);

    const [isModalVisible, setIsModalVisible] = useState(false);
    const [isScanningActive, setIsScanningActive] = useState(false);
    const [isNewItem, setIsNewItem] = useState(false);

    const [allRayons, setAllRayons] = useState<string[]>([]);
    const [allAttributs, setAllAttributs] = useState<string[]>([]);
    const [allConservations, setAllConservations] = useState<string[]>([]);

    const [newPrice, setNewPrice] = useState('');
    const [selectedRayon, setSelectedRayon] = useState<string | undefined>();
    const [selectedAttribut, setSelectedAttribut] = useState<string | undefined>();

    const [selectedConservation, setSelectedConservation] = useState<string | undefined>();


    const showNotification = (msg: string, type: 'error' | 'success' = 'error') => {
        setNotification({ msg, type });
    };

    const handleManualSubmit = (barcode: string) => {
        setIsManualInputVisible(false);
        const mockResult = { data: barcode } as BarcodeScanningResult;
        handleBarcodeScanned(mockResult);
    };

    const fetchOptions = async (
        endpoint: string,
        setter: React.Dispatch<React.SetStateAction<string[]>>
    ) => {
        try {
            const response = await fetch(`${API_BASE_URL}/items/options/${endpoint}`);
            if (!response.ok) throw new Error(`Erreur réseau pour ${endpoint}`);
            const data: string[] = await response.json();
            setter(data);
        } catch (error) {
            showNotification(`Erreur fetchOoptions (${endpoint}): ${error}`, 'error')
        }
    };

    useEffect(() => {
        fetchOptions('rayons', setAllRayons);
        fetchOptions('attributs', setAllAttributs);
        fetchOptions('conservations', setAllConservations);
    }, []);

    if (!permission) {
        return (
            <View style={styles.container}>
                <ActivityIndicator size="large" color="#fff" />
            </View>
        );
    }

    if (!permission.granted) {
        return (
            <View style={styles.container}>
                <Text style={styles.permissionText}>Permission caméra nécessaire pour le scan</Text>
                <Button onPress={requestPermission} title="Accorder la permission" />

                <View style={styles.manualContainerBlocked}>
                    <Text style={styles.separatorText}>--- OU ---</Text>
                    <ManualItemInput onSubmit={handleManualSubmit} visible={true} />
                </View>
            </View>
        );
    }

    const handleBarcodeScanned = async (scanningResult: BarcodeScanningResult) => {
        if (scanned) return;
        setIsScanningActive(false);
        setScanned(true);
        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/${scanningResult.data}`);
            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Produit non trouvé");
            }

            const fetchedItem: ScannedItem = await response.json();

            const isNew = fetchedItem.rayon === null || (fetchedItem.price as number) === 0;
            setIsNewItem(isNew);
            setItem(fetchedItem);

            if (isNew) {
                setNewPrice(fetchedItem.price ? fetchedItem.price.toString() : '');
                setSelectedRayon(fetchedItem.rayon ? fetchedItem.rayon.toString() : '');
                setSelectedAttribut(fetchedItem.attribut ? fetchedItem.attribut.toString() : '');
                setSelectedConservation(fetchedItem.conservation ? fetchedItem.conservation.toString() : '');
                setIsModalVisible(true);
            } else {

                const existingItemObject = new Item(
                    fetchedItem.item_id as number,
                    fetchedItem.barcode_id as string,
                    fetchedItem.product_name as string,
                    fetchedItem.image_url as string,
                    fetchedItem.small_image_url as string,
                    fetchedItem.brands as string,
                    fetchedItem.nutriscore_grade as string,
                    fetchedItem.product_quantity as number,
                    fetchedItem.product_quantity_unit as string,
                    fetchedItem.ingredients_text as string,
                    fetchedItem.allergens_imported as string,
                    fetchedItem.price as number,
                    fetchedItem.rayon as string,
                    fetchedItem.attribut as string,
                    fetchedItem.conservation as string
                );
                showAddToCartConfirmation(existingItemObject, showNotification, () => {
                    setScanned(false);
                });
            }

        } catch (error) {
            showNotification(`Impossible de récupérer les détails pour ${scanningResult.data}. ${error}`, 'error');
            setScanned(false);
        } finally {
            setIsLoading(false);
        }
    };

    const handleCloseModal = () => {
        setIsModalVisible(false);
        setItem(null);
        setScanned(false);
        setNewPrice('');
        setSelectedRayon(undefined);
        setSelectedAttribut(undefined);
        setSelectedConservation(undefined);
    };

    const handleSubmitUpdates = async () => {
        if (!item) return;

        if (!isNewItem) {
            handleCloseModal();
            showNotification("Erreur: Tentative de modification d'un article existant. Opération annulée.", 'error');
            return;
        }

        const priceValue = parseFloat(newPrice.replace(',', '.'));
        if (isNaN(priceValue) || priceValue <= 0) {
            showNotification("Le prix doit être un nombre valide et supérieur à 0.", 'error')
            return;
        }

        if (!selectedRayon) {
            showNotification("Le 'Rayon' est obligatoire.", 'error')
            return;
        }
        if (!selectedAttribut) {
            showNotification("L''Attribut' est obligatoire.", 'error')
            return;
        }
        if (!selectedConservation) {
            showNotification("La conservation est obligatoire", 'error')
            return;
        }

        const updateDetails = {
            price: priceValue,
            rayon: selectedRayon,
            attribut: selectedAttribut,
            conservation: selectedConservation,
        };

        setIsLoading(true);
        try {
            const response = await fetch(`${API_BASE_URL}/items/barcode/details/${item.barcode_id}`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(updateDetails),
            });

            if (!response.ok) {
                const err = await response.json();
                throw new Error(err.detail || "Erreur lors de la mise à jour");
            }

            const updatedItemObject = new Item(
                item.item_id as number,
                item.barcode_id as string,
                item.product_name as string,
                item.image_url as string,
                item.small_image_url as string,
                item.brands as string,
                item.nutriscore_grade as string,
                item.product_quantity as number,
                item.product_quantity_unit as string,
                item.ingredients_text as string,
                item.allergens_imported as string,
                priceValue,
                selectedRayon as string,
                selectedAttribut as string,
                selectedConservation as string
            );

            handleCloseModal();

            setTimeout(() => {
                showAddToCartConfirmation(updatedItemObject, showNotification, () => {
                    setScanned(false);
                });
            }, 500);

        } catch (error) {
            showNotification(`Échec de la mise à jour: ${error}`, 'error')
        } finally {
            setIsLoading(false);
        }
    };

    const renderStartButton = () => {
        if (!isScanningActive && !isModalVisible && !isLoading && !isManualInputVisible) {
            return (
                <View style={styles.buttonsWrapper}>
                    <TouchableOpacity
                        style={[styles.actionButton, styles.scanButton]}
                        onPress={() => setIsScanningActive(true)}
                    >
                        <Text style={styles.actionButtonText}>Scanner</Text>
                    </TouchableOpacity>

                    <TouchableOpacity
                        style={[styles.actionButton, styles.manualButton]}
                        onPress={() => setIsManualInputVisible(true)}
                    >
                        <Text style={styles.actionButtonText}>Saisie Manuelle</Text>
                    </TouchableOpacity>
                </View>
            );
        }
        return null;
    };

    return (
        <View style={styles.container}>
            <CameraView
                style={styles.camera}
                facing="back"
                onBarcodeScanned={isScanningActive && !scanned ? handleBarcodeScanned : undefined}
                barcodeScannerSettings={{
                    barcodeTypes: ['ean13', 'qr'],
                }}
            />

            {isLoading && !isModalVisible &&
                <ActivityIndicator size="large" color="#fff" style={styles.loading} />
            }

            {renderStartButton()}

            <Modal
                visible={isManualInputVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={() => setIsManualInputVisible(false)}
            >
                <View style={styles.modalContainer}>
                    <ManualItemInput
                        onSubmit={handleManualSubmit}
                        onCancel={() => setIsManualInputVisible(false)}
                        visible={true}
                    />
                </View>
            </Modal>

            <Modal
                visible={isModalVisible}
                animationType="slide"
                transparent={true}
                onRequestClose={handleCloseModal}
            >
                <View style={styles.modalContainer}>

                    <View style={styles.modalView}>

                        <ScrollView contentContainerStyle={styles.modalScrollContent}>
                            {item && (
                                <>
                                    <Text style={styles.modalTitle}>
                                        {"Nouvel Article"}
                                    </Text>

                                    <Image
                                        style={styles.itemImage}
                                        source={{ uri: item.small_image_url || undefined }}
                                        resizeMode="contain"
                                    />

                                    <Text style={styles.itemTextName}>{item.product_name}</Text>

                                    <Text style={styles.label}>Prix (€)</Text>
                                    <TextInput
                                        style={styles.input}
                                        keyboardType="numeric"
                                        value={newPrice}
                                        onChangeText={setNewPrice}
                                        placeholder="0.00"
                                        placeholderTextColor="#999"
                                    />

                                    <AppPicker
                                        label="Rayon"
                                        selectedValue={selectedRayon}
                                        onValueChange={setSelectedRayon}
                                        items={allRayons}
                                    />

                                    <AppPicker
                                        label="Attribut"
                                        selectedValue={selectedAttribut}
                                        onValueChange={setSelectedAttribut}
                                        items={allAttributs}
                                    />

                                    <AppPicker
                                        label="Conservation"
                                        selectedValue={selectedConservation}
                                        onValueChange={setSelectedConservation}
                                        items={allConservations}
                                    />

                                    <View style={styles.buttonContainer}>
                                        <Button title="Annuler" onPress={handleCloseModal} color="red" />

                                        {isLoading && isModalVisible ? (
                                            <ActivityIndicator style={{ paddingHorizontal: 20 }} />
                                        ) : (
                                            <Button title="Valider" onPress={handleSubmitUpdates} />
                                        )}
                                    </View>
                                </>
                            )}
                        </ScrollView>
                    </View>
                </View>
            </Modal>
            <ErrorNotification
                message={notification?.msg || null}
                type={notification?.type}
                onClose={() => setNotification(null)}
            />
        </View>
    );
}


const styles = StyleSheet.create({
    startScanButtonWrapper: {
        position: 'absolute',
        bottom: 40,
        width: '80%',
        alignSelf: 'center',
    },
    startScanButton: {
        backgroundColor: '#007bff',
        paddingVertical: 15,
        borderRadius: 25,
        alignItems: 'center',
        justifyContent: 'center',
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 4 },
        shadowOpacity: 0.3,
        shadowRadius: 5,
        elevation: 8,
    },
    buttonsWrapper: {
        position: 'absolute',
        bottom: 50,
        width: '100%',
        flexDirection: 'row',
        justifyContent: 'space-between',
        paddingHorizontal: 20,
        gap: 15,
    },
    actionButton: {
        flex: 1,
        paddingVertical: 18,
        borderRadius: 12,
        alignItems: 'center',
        justifyContent: 'center',
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 4 },
        shadowOpacity: 0.3,
        shadowRadius: 4,
        elevation: 6,
    },
    scanButton: {
        backgroundColor: '#007bff',
    },
    manualButton: {
        backgroundColor: '#495057',
    },
    actionButtonText: {
        color: 'white',
        fontSize: 16,
        fontWeight: 'bold',
        letterSpacing: 0.5,
    },
    startScanButtonText: {
        color: 'white',
        fontSize: 18,
        fontWeight: 'bold',
    },
    manualContainerBlocked: {
        width: '100%',
        marginTop: 30,
        alignItems: 'center',
    },
    separatorText: {
        color: 'white',
        marginVertical: 10,
        fontWeight: 'bold',
    },
    container: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: '#000',
    },
    permissionText: {
        color: 'white',
        textAlign: 'center',
        margin: 20,
    },
    camera: {
        ...StyleSheet.absoluteFillObject,
    },
    loading: {
        position: 'absolute',
    },

    modalContainer: {
        flex: 1,
        justifyContent: 'center',
        alignItems: 'center',
        backgroundColor: 'rgba(0,0,0,0.6)',
    },
    modalView: {
        width: '90%',
        maxHeight: '85%',
        backgroundColor: 'white',
        borderRadius: 20,
        padding: 20,
        shadowColor: '#000',
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.25,
        shadowRadius: 4,
        elevation: 5,
    },
    modalScrollContent: {
        alignItems: 'center',
    },

    modalTitle: {
        fontSize: 22,
        fontWeight: 'bold',
        marginBottom: 15,
        color: '#111',
    },
    itemImage: {
        width: 150,
        height: 150,
        marginBottom: 15,
        borderRadius: 8,
    },
    itemTextName: {
        fontSize: 18,
        fontWeight: '600',
        textAlign: 'center',
        marginBottom: 20,
    },

    label: {
        fontWeight: 'bold',
        marginTop: 10,
        color: '#333',
        width: '100%'
    },
    input: {
        width: '100%',
        height: 50,
        borderColor: '#ccc',
        borderWidth: 1,
        borderRadius: 8,
        paddingHorizontal: 15,
        marginBottom: 10,
        marginTop: 5,
        backgroundColor: '#f9f9f9',
        fontSize: 16,
    },

    buttonContainer: {
        flexDirection: 'row',
        justifyContent: 'space-around',
        width: '100%',
        marginTop: 25,
        marginBottom: 10,
    }
});