import React, { useEffect, useState } from 'react';
import { View, Text, StyleSheet, ScrollView, Alert, ActivityIndicator, TouchableOpacity } from 'react-native';
import { useStripe } from "@stripe/stripe-react-native";
import { CartService } from '../services/CartService';
import { CartItem } from '../types/CartItem';
import { API_BASE_URL } from "../config";
import { useNavigation } from '@react-navigation/native';
import { customer_ID } from '../config';

export default function RecapScreen() {
    const [cartItems, setCartItems] = useState<CartItem[]>([]);
    const [loading, setLoading] = useState(false);
    const { initPaymentSheet, presentPaymentSheet } = useStripe();
    const navigation = useNavigation();

    const userId = customer_ID;

    useEffect(() => {
        const items = CartService.getCartItems();
        setCartItems(items);
        initializePaymentSheet(items);
    }, []);

    const totalHT = CartService.getTotalPrice() as number;
    const tvaAmount = totalHT * 0.20;
    const totalTTC = totalHT + tvaAmount;

    const fetchPaymentSheetParams = async (itemsToPay: CartItem[]) => {
        const pendingItems = itemsToPay.map(ci => ({
            id: ci.getItem().getId(),
            amount: ci.getQuantity()
        }));

        if (pendingItems.length === 0) return null;

        try {
            const response = await fetch(`${API_BASE_URL}/payments/`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    "pending_items": pendingItems,
                    "customer_id": userId
                })
            });

            if (!response.ok) {
                console.error("Erreur backend paiement", await response.text());
                return null;
            }

            const { paymentIntent, ephemeralKey, customer } = await response.json();
            return { paymentIntent, ephemeralKey, customer };
        } catch (error) {
            console.error("Erreur réseau paiement", error);
            return null;
        }
    };

    const initializePaymentSheet = async (itemsToPay: CartItem[]) => {
        setLoading(true);
        const params = await fetchPaymentSheetParams(itemsToPay);

        if (!params) {
            Alert.alert("Erreur", "Impossible de préparer le paiement.");
            setLoading(false);
            return;
        }

        const { paymentIntent, ephemeralKey, customer } = params;

        const { error } = await initPaymentSheet({
            merchantDisplayName: "Mon Supermarché",
            customerId: customer,
            customerEphemeralKeySecret: ephemeralKey,
            paymentIntentClientSecret: paymentIntent,
            allowsDelayedPaymentMethods: false,
            defaultBillingDetails: {
                name: 'Client Test',
            }
        });

        if (error) {
            console.log("Erreur init Stripe", error);
        }
        setLoading(false);
    };

    const openPaymentSheet = async () => {
        const { error } = await presentPaymentSheet();

        if (error) {
            Alert.alert(`Paiement échoué`, error.message);
        } else {
            Alert.alert('Succès', 'Votre commande est validée !');

            CartService.clearCart();
            setCartItems([]);

            navigation.goBack();
        }
    };

    const formatPrice = (price: number) => {
        return new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR' }).format(price);
    };

    return (
        <View style={styles.container}>
            <ScrollView style={styles.scrollContent}>
                <Text style={styles.title}>Facture</Text>

                <View style={styles.invoiceCard}>
                    <View style={styles.rowHeader}>
                        <Text style={[styles.colProduct, styles.headerText]}>Produit</Text>
                        <Text style={[styles.colQty, styles.headerText]}>Qté</Text>
                        <Text style={[styles.colTotal, styles.headerText]}>Prix</Text>
                    </View>
                    <View style={styles.divider} />

                    {cartItems.map((ci, index) => (
                        <View key={index} style={styles.rowItem}>
                            <Text style={styles.colProduct} numberOfLines={1}>
                                {ci.getItem().getProductName()}
                            </Text>
                            <Text style={styles.colQty}>x{ci.getQuantity()}</Text>
                            <Text style={styles.colTotal}>
                                {(ci.getTotalPrice() as number).toFixed(2)} €
                            </Text>
                        </View>
                    ))}

                    <View style={[styles.divider, { marginVertical: 15 }]} />

                    <View style={styles.summaryRow}>
                        <Text style={styles.summaryLabel}>Total HT</Text>
                        <Text style={styles.summaryValue}>{formatPrice(totalHT)}</Text>
                    </View>
                    <View style={styles.summaryRow}>
                        <Text style={styles.summaryLabel}>TVA (20%)</Text>
                        <Text style={styles.summaryValue}>{formatPrice(tvaAmount)}</Text>
                    </View>

                    <View style={styles.totalRow}>
                        <Text style={styles.totalLabel}>Total TTC</Text>
                        <Text style={styles.totalValue}>{formatPrice(totalTTC)}</Text>
                    </View>
                </View>
            </ScrollView>

            <View style={styles.footer}>
                <TouchableOpacity
                    style={[styles.payButton, (loading || cartItems.length === 0) && styles.disabledButton]}
                    onPress={openPaymentSheet}
                    disabled={loading || cartItems.length === 0}
                >
                    {loading ? (
                        <ActivityIndicator color="#fff" />
                    ) : (
                        <Text style={styles.payButtonText}>Payer {formatPrice(totalTTC)}</Text>
                    )}
                </TouchableOpacity>
            </View>
        </View>
    );
}

const styles = StyleSheet.create({
    container: {
        flex: 1,
        backgroundColor: '#F2F2F7'
    },

    scrollContent: {
        padding: 20
    },

    title: {
        fontSize: 28,
        fontWeight: '800',
        marginBottom: 20,
        color: '#000'
    },

    invoiceCard: {
        backgroundColor: '#fff',
        borderRadius: 16,
        padding: 20,
        shadowColor: "#000",
        shadowOffset: { width: 0, height: 2 },
        shadowOpacity: 0.05,
        shadowRadius: 8,
        elevation: 2,
        marginBottom: 50,
    },

    rowHeader: {
        flexDirection: 'row',
        marginBottom: 10
    },

    headerText: {
        color: '#8E8E93',
        fontSize: 12,
        fontWeight: '600',
        textTransform: 'uppercase'
    },

    rowItem: {
        flexDirection: 'row',
        paddingVertical: 12
    },

    colProduct: {
        flex: 1,
        fontSize: 15,
        color: '#000'
    },

    colQty: {
        width: 40,
        textAlign: 'center',
        fontSize: 15,
        color: '#8E8E93'
    },

    colTotal: {
        width: 70,
        textAlign: 'right',
        fontSize: 15,
        fontWeight: '600',
        color: '#000'
    },

    divider: {
        height: 1,
        backgroundColor: '#E5E5EA'
    },

    summaryRow: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginBottom: 8
    },

    summaryLabel: {
        fontSize: 15,
        color: '#8E8E93'
    },

    summaryValue: {
        fontSize: 15,
        color: '#000'
    },

    totalRow: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginTop: 10,
        paddingTop: 10,
        borderTopWidth: 1,
        borderColor: '#E5E5EA'
    },

    totalLabel: {
        fontSize: 18,
        fontWeight: '700',
        color: '#000'
    },

    totalValue: {
        fontSize: 22,
        fontWeight: '800',
        color: '#007AFF'
    },

    footer: {
        padding: 20,
        backgroundColor: '#fff',
        borderTopWidth: 1,
        borderColor: '#E5E5EA'
    },

    payButton: {
        backgroundColor: '#34C759',
        borderRadius: 14,
        paddingVertical: 16,
        alignItems: 'center',
    },

    disabledButton: {
        opacity: 0.6
    },

    payButtonText: {
        color: '#fff',
        fontSize: 18,
        fontWeight: '700'
    }
});