"use client";

import { useEffect, useMemo, useState } from "react";
import * as d3 from "d3";

const data = [
    { year: 2018, region: "IDF", value: 12 },
    { year: 2019, region: "IDF", value: 12.5 },
    { year: 2020, region: "IDF", value: 13 },
    { year: 2021, region: "IDF", value: 13.2 },
    { year: 2022, region: "IDF", value: 13.8 },

    { year: 2018, region: "NAQ", value: 9 },
    { year: 2019, region: "NAQ", value: 9.3 },
    { year: 2020, region: "NAQ", value: 9.7 },
    { year: 2021, region: "NAQ", value: 10.1 },
    { year: 2022, region: "NAQ", value: 10.4 },

    { year: 2018, region: "PAC", value: 8 },
    { year: 2019, region: "PAC", value: 8.1 },
    { year: 2020, region: "PAC", value: 8.4 },
    { year: 2021, region: "PAC", value: 8.9 },
    { year: 2022, region: "PAC", value: 9.2 },
];


const regions = ["IDF", "NAQ", "PAC"];

export default function Temporelle() {
    const [region, setRegion] = useState<string | null>(null);
    const [startYear, setStartYear] = useState<number>(2018);
    const [endYear, setEndYear] = useState<number>(2022);

    const allYears = [2018, 2019, 2020, 2021, 2022];

    const [minYear, setMinYear] = useState<number>(2018);
    const [maxYear, setMaxYear] = useState<number>(2022);

    const width = 900;
    const height = 600;
    const margin = { top: 40, right: 30, bottom: 40, left: 60 };


    useEffect(() => {
        if (minYear > maxYear) {
            setMaxYear(minYear);
        }
    }, [minYear]);


    /**
     * DATA TRANSFORM
     */
    const filteredData = useMemo(() => {
        return data.filter(d =>
            d.year >= minYear &&
            d.year <= maxYear &&
            (region ? d.region === region : true)
        );
    }, [minYear, maxYear, region]);



    const groupedData = useMemo(() => {
        return d3.group(filteredData, d => d.region);
    }, [filteredData]);

    // logique années
    const startYears = useMemo(() => {
        return allYears.filter(y => y <= maxYear);
    }, [maxYear]);

    const endYears = useMemo(() => {
        return allYears.filter(y => y >= minYear);
    }, [minYear]);

    /**
     * SCALES
     */
    const xScale = useMemo(() => {
        return d3.scaleLinear()
            .domain([minYear, maxYear])
            .range([margin.left, width - margin.right]);
    }, [minYear, maxYear]);


    const yScale = useMemo(() => {
        return d3.scaleLinear()
            .domain([
                0,
                d3.max(data, d => d.value)! + 1
            ])
            .nice()
            .range([height - margin.bottom, margin.top]);
    }, []);

    const colorScale = d3.scaleOrdinal<string>()
        .domain(regions)
        .range(["#e63946", "#457b9d", "#2a9d8f"]);

    /**
     * === LINE GENERATOR ================================
     */
    const line = d3.line<any>()
        .x(d => xScale(d.year))
        .y(d => yScale(d.value))
        .curve(d3.curveMonotoneX);

    /**
     * === AXES TICKS ====================================
     */
    const xTicks = xScale.ticks(5);
    const yTicks = yScale.ticks(5);

    return (
        <section className="w-full max-w-4xl flex">

            {/* === CONTROLS =============================== */}
            <div className="rounded-xl bg-white shadow-sm p-6">
                <h2>Taux d'imposition moyen par région</h2>

                <label className="text-sm font-medium text-gray-700">
                    Région
                </label>

                <select
                    onChange={e => setRegion(e.target.value || null)}
                    className="rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-700 shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500">
                    <option value="">Toutes les régions</option>
                    {regions.map(r => (
                        <option key={r} value={r}>{r}</option>
                    ))}
                </select>

                <label className="text-sm font-medium text-gray-700">
                    De
                </label>

                {/* Année de début */}
                <select
                    value={minYear}
                    onChange={e => setMinYear(Number(e.target.value))}
                    className="rounded-md border border-gray-300 px-3 py-2 text-sm"
                >
                    {startYears.map(year => (
                        <option key={year + "start"} value={year}>
                            {year}
                        </option>
                    ))}
                </select>

                <label className="text-sm font-medium text-gray-700">
                    à
                </label>

                {/* Année de fin */}
                <select
                    value={maxYear}
                    onChange={e => setMaxYear(Number(e.target.value))}
                    className="rounded-md border border-gray-300 px-3 py-2 text-sm"
                >
                    {endYears.map(year => (
                        <option key={year + "end"} value={year}>
                            {year}
                        </option>
                    ))}
                </select>


                {/* === SVG =================================== */}
                <svg width={width} height={height}>

                    {/* Grid */}
                    {yTicks.map(tick => (
                        <line
                            key={tick}
                            x1={margin.left}
                            x2={width - margin.right}
                            y1={yScale(tick)}
                            y2={yScale(tick)}
                            stroke="#eee"
                        />
                    ))}

                    {/* Axes */}
                    <line
                        x1={margin.left}
                        x2={margin.left}
                        y1={margin.top}
                        y2={height - margin.bottom}
                        stroke="#999"
                    />
                    <line
                        x1={margin.left}
                        x2={width - margin.right}
                        y1={height - margin.bottom}
                        y2={height - margin.bottom}
                        stroke="#999"
                    />

                    {/* Y labels */}
                    {yTicks.map(tick => (
                        <text
                            key={tick}
                            x={margin.left - 10}
                            y={yScale(tick)}
                            textAnchor="end"
                            alignmentBaseline="middle"
                            fontSize="10"
                            fill="#666"
                        >
                            {tick} %
                        </text>
                    ))}

                    {/* X labels */}
                    {xTicks.map(tick => (
                        <text
                            key={tick}
                            x={xScale(tick)}
                            y={height - margin.bottom + 20}
                            textAnchor="middle"
                            fontSize="10"
                            fill="#666"
                        >
                            {tick}
                        </text>
                    ))}

                    {/* Lines */}
                    {[...groupedData.entries()].map(([region, values]) => (
                        <path
                            key={region}
                            d={line(values)!}
                            fill="none"
                            stroke={colorScale(region)}
                            strokeWidth={2.5}
                        />
                    ))}

                    {/* Points */}
                    {[...groupedData.entries()].flatMap(([region, values]) =>
                        values.map(d => (
                            <circle
                                key={`${region}-${d.year}`}
                                cx={xScale(d.year)}
                                cy={yScale(d.value)}
                                r={4}
                                fill={colorScale(region)}
                                className="opacity-90"
                            />
                        ))
                    )}
                </svg>
            </div>
        </section>
    );
}
