"use client";

/*
    TODO : 
        - séparer svg du code
*/

import { useEffect, useMemo, useState } from "react";
import * as d3 from "d3";
import {taxes} from '@/data/taxes';
import { taxeStats } from "@/type/TaxeStats";


export default function Temporelle() {
    const [taxe, setTaxe] = useState("cves");
    const allYears = [2018, 2019, 2020, 2021, 2022];

    const [minYear, setMinYear] = useState<number>(2018);
    const [maxYear, setMaxYear] = useState<number>(2022);

    const [isLoading, setIsLoading] = useState<boolean>(false);

    const [data, setData] = useState<taxeStats[]>([]);

    const [activeRegion, setActiveRegion] = useState<string | null>(null);


    const width = 900;
    const height = 600;
    const margin = { top: 40, right: 30, bottom: 40, left: 60 };


    useEffect(() => {
        if (minYear > maxYear) {
            setMaxYear(minYear);
        }
    }, [minYear]);

    useEffect(() => {
        if (!taxe) return;

        const fetchData = async () => {
            setIsLoading(true);
            console.log("Fetching data for", taxe, minYear, maxYear);

            try {
                const results: taxeStats[] = [];

                for (let year = minYear; year <= maxYear; year++) {
                    const res = await fetch(
                        `https://localhost/${taxe}/stats?annee=${year}&groupBy=region&metric=taux`
                    );

                    if (!res.ok) {
                        throw new Error(`Erreur API année ${year}`);
                    }

                    const apiData: any = await res.json();
                    console.log(`Année ${year}`, apiData.member);

                    apiData.member.forEach((d: any) => {
                        results.push({
                            region: d.label,
                            year,
                            value: d.value,
                        });
                    });
                }

                setData(results);
            } catch (err) {
                console.error("Erreur fetch:", err);
            } finally {
                setIsLoading(false);
            }
        };

        fetchData();
    }, [taxe, minYear, maxYear]);

    /**
     * Filtre Data
     */
    const filteredData = useMemo(() => {
        return data.filter(d =>
            d.year >= minYear &&
            d.year <= maxYear
        );
    }, [data, minYear, maxYear]);


    const groupedData = useMemo(() => {
        return d3.group(filteredData, d => d.region);
    }, [filteredData]);

    // logique années
    const startYears = useMemo(() => {
        return allYears.filter(y => y <= maxYear);
    }, [maxYear]);

    const endYears = useMemo(() => {
        return allYears.filter(y => y >= minYear);
    }, [minYear]);

    const yearsRange = useMemo(() => {
        return d3.range(minYear, maxYear + 1);
    }, [minYear, maxYear]);

    const xScale = useMemo(() => {
        return d3.scalePoint<number>()
            .domain(yearsRange)
            .range([margin.left, width - margin.right]);
    }, [yearsRange]);


    const xTicks = yearsRange;

    const yMax = d3.max(data, d => d.value) ?? 0;

    const yScale = useMemo(() => {
        return d3.scaleLinear()
            .domain([0, yMax + 1])
            .nice()
            .range([height - margin.bottom, margin.top]);
    }, [yMax]);


    const yTicks = yScale.ticks(5);

    const regions = useMemo(() => {
        return Array.from(new Set(data.map(d => d.region)));
    }, [data]);


    const colorScale = useMemo(() => {
        return d3.scaleOrdinal<string>()
            .domain(regions)
            .range(d3.schemeTableau10);
    }, [regions]);

    const line = d3.line<any>()
        .x(d => xScale(d.year)!)
        .y(d => yScale(d.value))
        .curve(d3.curveMonotoneX);


    /**
     * AXES TICKS 
     */

    return (
        <section className="w-full max-w-4xl flex">

            {/* CONTROLS  */}
            <div className="rounded-xl bg-white text-gray-700 shadow-sm p-6">
                <h2>Taux d'imposition moyen par région</h2>

                <label className="text-sm font-medium text-gray-700">
                    Taxe
                </label>

                <select
                    onChange={e => setTaxe(e.target.value)}
                    className="rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-700 shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500">
                    {taxes.map(t => (
                        <option key={t.route} value={t.route}>{t.label}</option>
                    ))}
                </select>

                <label className="text-sm font-medium text-gray-700">
                    De
                </label>

                {/* Année de début */}
                <select
                    value={minYear}
                    onChange={e => setMinYear(Number(e.target.value))}
                    className="rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-700 shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                >
                    {startYears.map(year => (
                        <option key={year + "start"} value={year}>
                            {year}
                        </option>
                    ))}
                </select>

                <label className="text-sm font-medium text-gray-700">
                    à
                </label>

                {/* Année de fin */}
                <select
                    value={maxYear}
                    onChange={e => setMaxYear(Number(e.target.value))}
                    className="rounded-md border border-gray-300 bg-white px-3 py-2 text-sm text-gray-700 shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                >
                    {endYears.map(year => (
                        <option key={year + "end"} value={year}>
                            {year}
                        </option>
                    ))}
                </select>



                {/* SVG  */}
                <svg width={width} height={height}>
                    {/* Grid */}
                    {yTicks.map(tick => (
                        <line
                            key={tick}
                            x1={margin.left}
                            x2={width - margin.right}
                            y1={yScale(tick)}
                            y2={yScale(tick)}
                            stroke="#eee"
                        />
                    ))}

                    {/* Axes */}
                    <line
                        x1={margin.left}
                        x2={margin.left}
                        y1={margin.top}
                        y2={height - margin.bottom}
                        stroke="#999"
                    />
                    <line
                        x1={margin.left}
                        x2={width - margin.right}
                        y1={height - margin.bottom}
                        y2={height - margin.bottom}
                        stroke="#999"
                    />

                    {/* Y labels */}
                    {yTicks.map(tick => (
                        <text
                            key={tick}
                            x={margin.left - 10}
                            y={yScale(tick)}
                            textAnchor="end"
                            alignmentBaseline="middle"
                            fontSize="10"
                            fill="#666"
                        >
                            {tick} %
                        </text>
                    ))}

                    {/* X labels */}
                    {xTicks.map(tick => (
                        <text
                            key={tick}
                            x={xScale(tick)}
                            y={height - margin.bottom + 20}
                            textAnchor="middle"
                            fontSize="10"
                            fill="#666"
                        >
                            {tick}
                        </text>
                    ))}

                    {/* Lignes */}
                    {[...groupedData.entries()].map(([region, values]) => {
                        const isActive =
                            activeRegion === null || activeRegion === region;

                        return (
                            <path
                                key={region}
                                d={line(values)!}
                                fill="none"
                                stroke={colorScale(region)}
                                strokeWidth={isActive ? 3 : 1.5}
                                opacity={isActive ? 1 : 0.2}
                                onMouseEnter={() => setActiveRegion(region)}
                                onMouseLeave={() => setActiveRegion(null)}
                                className="transition-all duration-200"
                            />
                        );
                    })}

                    {/* Points */}
                    {[...groupedData.entries()].flatMap(([region, values]) =>
                        values.map(d => {
                            const isActive =
                                activeRegion === null || activeRegion === region;

                            return (
                                <circle
                                    key={`${region}-${d.year}`}
                                    cx={xScale(d.year)}
                                    cy={yScale(d.value)}
                                    r={isActive ? 4 : 3}
                                    fill={colorScale(region)}
                                    opacity={isActive ? 1 : 0.3}
                                />
                            );
                        })
                    )}
                </svg>

                {/* Légende */}

                <div className="mt-6 flex flex-wrap gap-4 justify-center">
                    {regions.map(region => {
                        const isActive =
                            activeRegion === null || activeRegion === region;

                        return (
                            <div
                                key={region}
                                className="flex items-center gap-2 cursor-pointer transition-opacity"
                                onMouseEnter={() => setActiveRegion(region)}
                                onMouseLeave={() => setActiveRegion(null)}
                                onClick={() =>
                                    setActiveRegion(activeRegion === region ? null : region)
                                }
                                style={{ opacity: isActive ? 1 : 0.4 }}
                            >
                                <span
                                    className="w-3 h-3 rounded-sm"
                                    style={{ backgroundColor: colorScale(region) }}
                                />

                                <span className="text-sm text-gray-700">
                                    
                                    {region}
                                </span>
                            </div>
                        );
                    })}
                </div>

            </div>
        </section >
    );
}
