<?php 

namespace App\Command;

use App\Entity\Tfpnb;
use App\Entity\Tfpb;
use App\Entity\Th;
use App\Entity\Cfe;
use Doctrine\ORM\EntityManagerInterface;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Stopwatch\Stopwatch;

#[AsCommand(name: 'app:import-xlsx', description: 'Importation REI ultra-rapide via CSV')]
class ImportDataCommand extends Command
{
    private const MAPPING_DICTIONARY = [
        'DEP' => 'DEPT', 'COM' => 'COMMUNE',
        'A11' => 'TFPNB_TAUX', 'A12' => 'TFPNB_MONTANT',
        'E12' => 'TFPB_TAUX', 'E13' => 'TFPB_MONTANT',
        'H12' => 'TH_TAUX', 'H13' => 'TH_MONTANT',
        'P12' => 'CFE_TAUX', 'P13' => 'CFE_MONTANT',
    ];

    private const ANNEES_A_TRAITER = [2022]; 

    public function __construct(private EntityManagerInterface $em) {
        parent::__construct();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $stopwatch = new Stopwatch();

        $stopwatch->start('import-global');
        
        $this->em->getConnection()->getConfiguration()->setMiddlewares([]);

        foreach (self::ANNEES_A_TRAITER as $annee) {
            $stopwatch->start('annee-' . $annee);
            $io->section("Importation de l'année $annee");

            $mappingFile = "data/mapping_$annee.xlsx";
            $dataFile = "data/REI_$annee.csv";

            if (!file_exists($mappingFile) || !file_exists($dataFile)) {
                $io->error("Fichiers manquants pour $annee");
                continue;
            }

            $colMapping = $this->loadYearlyMapping($mappingFile);

            if (($handle = fopen($dataFile, "r")) !== FALSE) {
                $io->progressStart();
                $index = 0;

                while (($rowData = fgetcsv($handle, 0, ";", '"', "\\")) !== FALSE) {
                    $index++;
                    if ($index === 1) continue;

                    $this->createEntry(Tfpnb::class, $annee, $rowData, $colMapping);
                    $this->createEntry(Tfpb::class, $annee, $rowData, $colMapping);
                    $this->createEntry(Th::class, $annee, $rowData, $colMapping);
                    $this->createEntry(Cfe::class, $annee, $rowData, $colMapping);

                    if ($index % 500 === 0) {
                        $this->em->flush();
                        $this->em->clear();
                        $io->progressAdvance(500);
                    }
                }
                
                $this->em->flush();
                $this->em->clear();
                fclose($handle);
                $io->progressFinish();
                $io->success("Année $annee terminée !");
            }
        }

        $globalEvent = $stopwatch->stop('import-global');
        $dureeTotale = number_format($globalEvent->getDuration() / 1000 / 60, 2);
        
        $io->success("Toutes les données ont été importées avec succès en $dureeTotale minutes !");

        return Command::SUCCESS;
    }

    private function loadYearlyMapping(string $filename): array
    {
        $spreadsheet = IOFactory::load($filename);
        $rows = $spreadsheet->getActiveSheet()->toArray();
        $mappingResult = [];
        foreach ($rows as $row) {
            $codeAdmin = trim((string)($row[1] ?? '')); 
            if (isset(self::MAPPING_DICTIONARY[$codeAdmin])) {
                $cleMetier = self::MAPPING_DICTIONARY[$codeAdmin];
                $mappingResult[$cleMetier] = (int)$row[0] - 1; 
            }
        }
        $spreadsheet->disconnectWorksheets();
        return $mappingResult;
    }

    private function createEntry(string $className, int $annee, array $rowData, array $mapping): void
    {
        $prefix = strtoupper((new \ReflectionClass($className))->getShortName());
        $idxTaux = $mapping[$prefix . '_TAUX'] ?? null;

        if ($idxTaux === null || !isset($rowData[$idxTaux]) || $rowData[$idxTaux] === '') {
            return;
        }

        $entity = new $className();
        $entity->setAnnee($annee);
        $entity->setDepartement((int)($rowData[$mapping['DEPT']] ?? 0));
        $entity->setNomCommune((string)($rowData[$mapping['COMMUNE']] ?? ''));

        $idxMontant = $mapping[$prefix . '_MONTANT'] ?? null;

        $taux = str_replace(',', '.', $rowData[$idxTaux]);
        $entity->setTauxNet((float)$taux);

        if ($idxMontant !== null && isset($rowData[$idxMontant])) {
            $montant = str_replace(',', '.', $rowData[$idxMontant]);
            $entity->setMontantReel((float)$montant);
        }

        $this->em->persist($entity);
    }
}