#include <mpi.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#define N 500
#define PUISSANCE 4

double top_debut, top_fin, temps;

int main(int argc, char* argv[])
{
    MPI_Init(&argc, &argv);
    int rank, P;
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
    MPI_Comm_size(MPI_COMM_WORLD, &P);
    MPI_Status status;
    int r = N / P;

	// Type bloc de lignes
    MPI_Datatype bloc_ligne;
    MPI_Type_contiguous(r * N, MPI_INT, &bloc_ligne);
    MPI_Type_commit(&bloc_ligne);

	// Type temporaire de bloc de colonnes 
	// Ce type n'est pas utilisable car les indices sont décalés
    MPI_Datatype bloc_colonne;
    MPI_Type_vector(N, r, N, MPI_INT, &bloc_colonne);
    MPI_Type_commit(&bloc_colonne);
	
	// Type bloc de colonnes
	// Avec MPI_type_create_resized, on corrige le décalage de bloc_colonne afin de rendre ce type contigu.
    MPI_Datatype bloc_col;
    MPI_Type_create_resized(bloc_colonne, 0, r * sizeof(int), &bloc_col);
    MPI_Type_commit(&bloc_col);

    int A[N][N], Ai[r][N], Cj[N][r], Ctemp[N][r], Ci[r][N], C[N][N];

    if (rank == 0) {
        srand(time(NULL));
        for (int i = 0; i < N; i++) {
            for (int j = 0; j < N; j++) {
                A[i][j] = rand() % 3;
                C[i][j] = A[i][j];
            }
		}
    }
    
    top_debut = MPI_Wtime();
	// on diffuse A à tous les processus sous forme de blocs de ligne
	// comme A ne change jamais, on ne le diffuse qu'une seule fois
    MPI_Scatter(A, 1, bloc_ligne, Ai, 1, bloc_ligne, 0, MPI_COMM_WORLD);
    
	// On calcule également le processus précédent et suivant dans l'anneau afin de permettre la communication des blocs.
	int suivant = (rank + 1) % P;
    int precedent = (rank - 1 + P) % P;

	// on itère sur l'indice de puissance - 1 (car la première itération calcule déjà A² !)
    for (int cp = 0; cp < PUISSANCE - 1; cp++) {
		// à chaque itération, on envoie la valeur de C sous forme de blocs de colonnes (stockés dans Cj)
		// C contient le résultat, il est donc important pour tous les processus de le garder à jour
        MPI_Scatter(C, 1, bloc_col, Cj, 1, bloc_ligne, 0, MPI_COMM_WORLD);

		// Calcul de Ci 
        for (int t = 0; t < P; t++) {
            int index = (rank - t + P) % P;
            for (int i = 0; i < r; i++) {
                for (int j = 0; j < r; j++) {
                    Ci[i][index * r + j] = 0;
                    for (int k = 0; k < N; k++)
                        Ci[i][index * r + j] += Ai[i][k] * Cj[k][j];
                }
            }

			// On envoie Cj sur les voisins de l'anneau
			// si le processus est de rang 0, on envoie avant de recevoir afin d'éviter tout blocage
            if (rank == 0) {
                MPI_Send(Cj, 1, bloc_ligne, suivant, 1, MPI_COMM_WORLD);
                MPI_Recv(Cj, 1, bloc_ligne, precedent, 1, MPI_COMM_WORLD, &status);
            }
            else {
                MPI_Recv(Ctemp, 1, bloc_ligne, precedent, 1, MPI_COMM_WORLD, &status);
                MPI_Send(Cj, 1, bloc_ligne, suivant, 1, MPI_COMM_WORLD);
                for (int i = 0; i < N; i++)
                    for (int k = 0; k < r; k++)
                        Cj[i][k] = Ctemp[i][k];
            }
        }

		// Enfin, on récupère la valeur calculée du bloc pour le stocker dans C.
        MPI_Gather(Ci, 1, bloc_ligne, C, 1, bloc_ligne, 0, MPI_COMM_WORLD);
    }

    top_fin = MPI_Wtime();
    temps = top_fin - top_debut;
    
    if (rank == 0) {
        printf("A = \n");
        for (int i = 0; i < N; i++) {
            printf("\t");
            for (int j = 0; j < N; j++)
                printf("%d ", A[i][j]);
            printf("\n");
        }
        printf("\n");
        printf("A%d= \n", PUISSANCE);
        for (int i = 0; i < N; i++) {
            printf("\t");
            for (int j = 0; j < N; j++)
                printf("%d ", C[i][j]);
            printf("\n");
        }

        printf("temps : %f \n\n", temps);
    }

    MPI_Finalize();
    return 0;
}

