const userModel = require('../models/user');

const chai = require('chai');
const chaiHttp = require('chai-http');
const htmlParser = require('node-html-parser');

// mocha needs it
// eslint-disable-next-line no-unused-vars
const should = chai.should();

process.env.NODE_ENV = 'test';
require('dotenv').config();
const server = require('../app');

chai.use(chaiHttp);

describe('Les utilisateurs', () => {
  before((done) => {
    userModel.User.deleteMany({}, (err) => {
      if (err) {
        console.error(err);
      }

      done();
    });
  });

  const clientData = {
    name: 'Client',
    role: 'client',
    email: 'client.test@test.test',
    password: '123mdr',
  };

  const agentData = {
    name: 'Agent',
    role: 'agent',
    email: 'agent.test@test.test',
    password: '123mdr',
  };

  // Requête de consultation des annonces
  describe('Action de création d\'utilisateur', () => {
    it('Supposément récupère la page de register', (done) => {
      chai.request(server)
          .get('/register')
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.be.html;
            res.should.have.status(200);
            done();
          });
    });

    it('Supposément crée un user client', (done) => {
      chai.request(server)
          .post('/register')
          .type('form')
          .send(clientData)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.have.status(200);

            userModel.User.findOne({email: clientData.email}).exec()
                .then((value) => chai.assert.isTrue(value !== null))
                .catch((err) => chai.assert.fail(err))
                .finally(() => done());
          });
    });

    it('Supposément crée un user agent', (done) => {
      chai.request(server)
          .post('/register')
          .type('form')
          .send(agentData)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.have.status(200);

            userModel.User.findOne({email: agentData.email}).exec()
                .then((value) => chai.assert.isTrue(value !== null))
                .catch((err) => chai.assert.fail(err))
                .finally(() => done());
          });
    });

    it('Supposément refuse de créer car email existe déjà', (done) => {
      chai.request.agent(server)
          .post('/register')
          .type('form')
          .send(agentData)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('div.msg-info') !== null);
            done();
          });
    });

    it('Supposément refuse de créer car email incorrect', (done) => {
      const vraiEmail = agentData.email;
      agentData.email = 'null';

      chai.request.agent(server)
          .post('/register')
          .type('form')
          .send(agentData)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('div.msg-info') !== null);

            agentData.email = vraiEmail;
            done();
          });
    });
  });

  describe('Actions d\'authentification', () => {
    it('Supposément récupère la page de login', (done) => {
      chai.request(server)
          .get('/login')
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.be.html;
            res.should.have.status(200);
            done();
          });
    });

    it('Supposément se connecte avec un user client', (done) => {
      chai.request(server)
          .post('/login')
          .send({
            username: clientData.email,
            password: clientData.password,
          })
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.have.status(200);
            done();
          });
    });

    it('Supposément plante car user n\'existe pas', (done) => {
      chai.request(server)
          .post('/login')
          .send({
            username: 'salut',
            password: 'c\'est cool',
          })
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            chai.assert.isTrue(res.redirects.length != 0);
            done();
          });
    });

    it('Supposément plante car le mot de passe est incorrect',
        (done) => {
          chai.request(server)
              .post('/login')
              .send({
                username: clientData.email,
                password: 'mickey01',
              })
              .end((err, res) => {
                if (err) {
                  chai.assert.fail(err);
                }

                chai.assert.isTrue(res.redirects.length != 0);
                done();
              });
        });

    it('Supposément se connecte avec un user agent', (done) => {
      const agent = chai.request.agent(server);

      agent
          .post('/login')
          .send({
            username: agentData.email,
            password: agentData.password,
          })
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            res.should.have.status(200);

            agent.get('/ads/create').end((err, res) => {
              if (err) {
                chai.assert.fail(err);
              }

              res.should.be.html;
              res.should.have.status(200);
              done();
            });
          });
    });

    it('Supposément plante car niveau de permission inexistant', (done) => {
      const authorizeUser = require('../src/permissions').authorizeUser;

      chai.assert.isFalse(authorizeUser({role: 'admin'}, 'gloubiboulga'));
      done();
    });

    it('Supposément se déconnecte', (done) => {
      const agent = chai.request.agent(server);

      agent
          .post('/login')
          .send({
            username: clientData.email,
            password: clientData.password,
          })
          .then(() => agent.get('/logout'))
          .then((res) => {
            res.should.be.html;
            res.should.be.ok;
            done();
          });
    });

    it('Supposément plante car droits insuffisants', (done) => {
      const agent = chai.request.agent(server);

      agent
          .post('/login')
          .send({
            username: clientData.email,
            password: clientData.password,
          })
          .then(() => {
            return agent
                .post('/ads/create')
                .send({});
          })
          .then((res) => {
            res.should.be.html;

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('.msg-error') != null);

            done();
          });
    });
  });
});
