const adModel = require('../models/ad');
const userModel = require('../models/user');

const chai = require('chai');
const chaiHttp = require('chai-http');
const htmlParser = require('node-html-parser');

// mocha needs it
// eslint-disable-next-line no-unused-vars
const should = chai.should();

process.env.NODE_ENV = 'test';
require('dotenv').config();
const server = require('../app');

chai.use(chaiHttp);

describe('Les annonces', () => {
  const agent = chai.request.agent(server);
  const agentData = {
    name: 'agent',
    role: 'agent',
    email: 'test@agent.test.fr',
    password: 'Test123!',
  };

  before(async () => {
    await adModel.Ad.deleteMany({}, (err) => {
      if (err) {
        console.error(err);
      }
    }).exec();

    await userModel.User.deleteMany({}, (err) => {
      if (err) {
        console.error(err);
      }
    }).exec();

    await agent.post('/register').send(agentData);

    await agent.post('/login').send({
      username: agentData.email,
      password: agentData.password,
    });
  });

  const formData = {
    title: 'Test',
    type: adModel.AD_TYPE.VALUE.RENTAL,
    transactionStatus: adModel.AD_TRANSACTIONSTATUS.VALUE.AVAILABLE,
    price: '420,69',
    published: false,
    pictures: [],
  };

  // Requêtes de création d'annonce
  describe('Actions de création des annonces', () => {
    it('Supposément obtient la page de création d\'annonce', (done) => {
      agent.get('/ads/create').end((err, res) => {
        if (err) {
          chai.assert.fail(err);
        }

        res.should.be.html;
        res.should.have.status(200);
        done();
      });
    });

    it('Supposément crée une annonce correcte', (done) => {
      agent
          .post('/ads/create')
          .type('form')
          .send(formData)
          .then((res) => {
            res.should.be.ok;
            return agent
                .post('/ads/create')
                .type('form')
                .send({...formData, published: 'on'});
          })
          .then((res) => {
            res.should.be.ok;
            return agent
                .post('/ads/create')
                .type('form')
                .send({...formData});
          })
          .then((res) => {
            res.should.be.ok;
            return agent
                .post('/ads/create')
                .type('form')
                .send({...formData, published: 'on'});
          })
          .then((res) => {
            res.should.be.ok;
            res.should.be.html;
            done();
          })
          .catch((err) => {
            chai.assert.fail(err);
          });
    });

    it(
        'Supposément rejette l\'annonce ' + 'car il lui manque des champs requis',
        (done) => {
          const formDataVide = {
            _method: 'put',
            title: 'Manque des champs obligatoires',
          };

          agent
              .post('/ads/create')
              .type('form')
              .send(formDataVide)
              .end((err, res) => {
                if (err) {
                  chai.assert.fail(err);
                }

                res.should.be.html;
                done();
              });
        },
    );

    it(
        'Supposément rejette l\'annonce ' + 'car le champ price pas conforme',
        (done) => {
          const formDataFaux = {
            title: 'Champ price pas conforme',
            type: adModel.AD_TYPE.VALUE.RENTAL,
            transactionStatus: adModel.AD_TRANSACTIONSTATUS.VALUE.AVAILABLE,
            price: '99999,969',
            published: 'on',
          };

          agent
              .post('/ads/create')
              .type('form')
              .send(formDataFaux)
              .end((err, res) => {
                if (err) {
                  chai.assert.fail(err);
                }

                res.should.be.html;
                res.should.have.status(406);
                done();
              });
        },
    );
  });

  // Requête de consultation des annonces
  describe('Action de listing des annonces', () => {
    it('Supposément obtient les annonces décrites ci-dessous', (done) => {
      adModel.Ad.find().then((ads) => {
        console.log(ads);
      });
      agent.get('/ads/').end((err, res) => {
        if (err) {
          chai.assert.fail(err);
        }

        res.should.be.html;
        res.should.have.status(200);
        done();
      });
    });
  });

  // Requêtes de mise à jour d'annonce
  describe('Action de mise à jour d\'annonce', () => {
    let adId;

    before(async () => {
      const ad = await adModel.Ad.findOne({});
      return adId = ad && ad.id;
    });

    it('Supposément fais bien la mise à jour', (done) => {
      agent.get(`/ads/update/${adId}`).end((err, res) => {
        if (err) {
          chai.assert.fail(err);
        }

        res.should.be.html;
        res.should.have.status(200);

        formData.title = 'Plus le même';
        formData['id'] = adId;

        agent
            .post('/ads/create')
            .send(formData)
            .end((err, res) => {
              if (err) {
                chai.assert.fail(err);
              }

              const html = htmlParser.parse(res.text);
              chai.assert.isTrue(
                  html.querySelector('div.msg-success') != null,
              );
              done();
            });
      });
    });

    it('Supposément plante vu qu\'une des données n\'est pas valide',
        (done) => {
          const wrongData = {
            ...formData,
            id: adId,
            price: 'abc',
          };

          agent
              .post(`/ads/create/`)
              .send(wrongData)
              .end((err, res) => {
                if (err) {
                  chai.assert.fail(err);
                }

                res.should.have.status(406);
                res.should.be.html;
                done();
              });
        });

    it('Supposément plante vu que l\'id existe pas en fait', (done) => {
      agent.get(`/ads/update/5`).end((err, res) => {
        if (err) {
          chai.assert.fail(err);
        }

        res.should.be.html;
        res.should.have.status(404);
        done();
      });
    });
  });

  // Requêtes liées aux questions/réponses
  describe('Actions liées aux questions/réponses', () => {
    const question = {
      body: 'Y\'a pas d\'arraignéees, t\'es sûr ?',
    };

    const reply = {
      body: 'Bonjour non y\'en a pas',
    };

    let adId; let questionId;

    before(async () => {
      const ad = await adModel.Ad.findOne({});
      adId = ad && ad.id;
    });

    it('Supposément crée une question correcte', (done) => {
      agent
          .post(`/ads/${adId}/questions/create`)
          .send(question)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            adModel.Ad.findOne({}).then((ad) => {
              chai.assert.isTrue(ad.questions[0] != null);
              chai.assert.equal(ad.questions[0].body, question.body);
              questionId = ad.questions[0]._id;
              done();
            });
          });
    });

    it('Supposément plante car question incorrecte', (done) => {
      question.body = null;

      agent
          .post(`/ads/${adId}/questions/create`)
          .send(question)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('div.msg-error') != null);

            done();
          });
    });

    it('Supposément crée une réponse correcte', (done) => {
      agent
          .post(`/ads/${adId}/questions/${questionId}/answers/create`)
          .send(reply)
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            adModel.Ad.findOne({}).then((ad) => {
              chai.assert.isTrue(ad.questions[0].answers[0] != null);
              done();
            });
          });
    });

    it('Supposément plante car réponse incorrecte', (done) => {
      agent
          .post(`/ads/${adId}/questions/${questionId}/answers/create`)
          .send({body: null})
          .end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('div.msg-error') != null);

            done();
          });
    });
  });

  describe('Action d\'affichage d\'annonce.s', () => {
    let adId;

    before(async () => {
      const ad = await adModel.Ad.findOne({});
      return adId = ad && ad.id;
    });

    it('Supposément plante car l\'image n\'existe pas', (done) => {
      agent
          .get(`/ads/${adId}/picture/1`)
          .then((res) => {
            res.should.be.html;
            res.should.have.status(500);
            done();
          });
    });

    it('Supposément affiche la page car elle existe', (done) => {
      agent
          .get(`/ads/draft/${adId}`)
          .then((res) => {
            res.should.be.ok;
            res.should.be.html;
            done();
          })
          .catch((err) => chai.assert.fail(err));
    });

    it('Supposément plante si elle n\'existe pas', (done) => {
      agent
          .get('/ads/qlsdfjlqskjdflqksjdf')
          .then((res) => {
            res.should.be.ok;
            res.should.be.html;
            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('.msg-log') != null);
            done();
          })
          .catch((err) => chai.assert.fail(err));
    });
  });

  // Requêtes de suppression d'annonce
  describe('Action de suppression d\'annonce', () => {
    it('Supposément plante vu que l\'id existe pas en fait', (done) => {
      agent.delete(`/ads/delete/aaaaaaaaaaaaaaaaaaaaaaaa`).end((err, res) => {
        if (err) {
          chai.assert.fail(err);
        }

        const html = htmlParser.parse(res.text);
        chai.assert.isTrue(html.querySelector('div.msg-info') != null);

        done();
      });
    });

    it('Supposément supprime bien vu que l\'id existe', (done) => {
      adModel.Ad.findOne({})
          .exec()
          .then((value) => {
            agent.delete(`/ads/delete/${value.id}`).end((err, res) => {
              if (err) {
                chai.assert.fail(err);
              }

              const html = htmlParser.parse(res.text);
              chai.assert.isTrue(html.querySelector('div.msg-success') != null);
            });
          })
          .catch((reason) => chai.assert.fail(reason))
          .finally(() => done());
    });

    it(
        'Supposément part en couille ' + 'vu que l\'id est pas sous le bon format',
        (done) => {
          agent.delete(`/ads/delete/5`).end((err, res) => {
            if (err) {
              chai.assert.fail(err);
            }

            const html = htmlParser.parse(res.text);
            chai.assert.isTrue(html.querySelector('div.msg-error') != null);

            done();
          });
        },
    );
  });
});


describe('Le site en général', () => {
  let agent;

  before((done) => {
    agent = chai.request.agent(server);
    done();
  });

  it('Supposément plante car route inexistante', (done) => {
    agent
        .get('/route/qui/nexiste/pas')
        .end((err, res) => {
          if (err) {
            chai.assert.fail(err);
          }

          res.should.have.status(404);
          res.should.be.html;
          done();
        });
  });
});
