const USER_PERMISSIONS = require('../models/user').USER_PERMISSIONS;

/**
 * Middleware de route qui ne laisse l'accès qu'aux utilisateurs
 * qualifiés par la condition donnée en paramètre.
 *
 * En cas d'autorisations insuffisantes l'utilisateur est redirigé vers la page
 * d'accueil avec un message d'erreur.
 * Ce middleware est à utiliser avant la véritable fonction de route.
 *
 * @param {string} level
 *    le niveau de permission requis pour accéder à la ressource
 * @return {Function}
 *    une fonction Express-friendly pour garder la chaîne intacte
 */
function authorize(level) {
  const requiredPermissionLevel = USER_PERMISSIONS[level];

  return function(req, res, next) {
    const requestEmitterPermissionLevel =
      USER_PERMISSIONS[(req.user && req.user.role) || 'NOBODY'];

    // User doesn't have required permissions to access this route
    if (requestEmitterPermissionLevel < requiredPermissionLevel) {
      req.flash('error', 'Vous n\'avez pas les autorisations requises' +
          ' pour accéder à cette ressource');
      res.redirect('/');
    } else {
      next();
    }
  };
}

/**
 * Vérifie les droits d'un utilisateur donné.
 *
 * @param {*} user
 *    l'utilisateur dont on vérifie les droits.
 * @param {string} level
 *    le niveau d'autorisation requis pour qu'il soit autorisé.
 * @return {boolean}
 *    vrai si l'utilisateur est autorisé.
 */
function authorizeUser(user, level) {
  const requiredPermissionLevel = USER_PERMISSIONS[level];
  const requestEmitterPermissionLevel =
    USER_PERMISSIONS[(user && user.role) || 'NOBODY'];

  if (!requiredPermissionLevel) {
    console.error(`checkUserPermissions: Le rôle ${level} n'existe pas !`);
    return false;
  }

  return requestEmitterPermissionLevel >= requiredPermissionLevel;
}

module.exports = {
  authorize,
  authorizeUser,
};
