const express = require('express');
const multer = require('multer');
const router = new express.Router();
const authorize = require('../src/permissions').authorize;

const adModel = require('../models/ad');

const upload = multer({
  storage: multer.memoryStorage(),
});

/* GET to get to get to the ad creation form */
router
    .get('/create', authorize('agent'), function(req, res) {
      res.render('ads/create', {title: 'Créer annonce'});
    })
    .post('/create', authorize('agent'), upload.array('pictures', 3), function(
        req,
        res,
    ) {
      const body = req.body;
      const id = body.id;

      const formData = {
        title: body.title,
        type: body.type,
        transactionStatus: body.transactionStatus,
        price: body.price.replace(',', '.'),
        published: body.published === 'on',
        description: body.description,
        availabilityDate:
        body.availabilityDate === '' ? null : body.availabilityDate,
      };

      if (req.files && req.files.length) {
        formData.pictures = req.files.map((f) => ({
          name: f.fieldName,
          body: f.buffer,
        }));
      }

      if (id) {
      // Peut-être charger l'objet en amont et le retourner si erreur ?
        adModel.Ad.updateOne({_id: id}, {$set: formData})
            .exec()
            .then(() => {
              req.flash('success', 'Mise à jour réussie !');
              res.redirect('/ads/');
            })
            .catch((reason) => {
              res.status(406).render('ads/create', {
                title: 'Modifier annonce',
                errors: reason.errors,
              });
            });
      } else {
        const newAd = new adModel.Ad(formData);

        newAd
            .save()
            .then(() => {
              req.flash('success', 'Création réussie !');
              res.redirect('/ads/');
            })
            .catch((reason) => {
              res.status(406).render('ads/create', {
                title: 'Créer annonce',
                errors: reason.errors,
              });
            });
      }
    })
    .get('/update/:id', authorize('agent'), function(req, res, next) {
      const id = req.params.id;

      adModel.Ad.findOne(
          {_id: id},
          {'pictures.body': 0, 'questions': 0},
          function(err, ad) {
            const errors = [];
            if (err) {
              errors.push(err.message);
            }
            if (!ad) {
              errors.push('L\'annonce cherchée n\'a pas été trouvée');
              res.status(404);
            } else {
              req.flash('success', 'Mise à jour réussie !');
            }

            res.render('ads/create', {
              title: 'Modifier annonce',
              ad: ad,
              errors_update: errors,
            });
          },
      );
    })
    .get('/delete/:id', authorize('agent'), deleteAdAction)
    .delete('/delete/:id', authorize('agent'), deleteAdAction)
    .post('/:id/questions/create', authorize('client'), function(req, res) {
      const question = {
        author: req.user.name,
        body: req.body.body,
        date: new Date(),
      };

      adModel.Ad.findByIdAndUpdate(req.params.id, {
        $push: {
          questions: question,
        },
      })
          .then((ad) => {
            req.flash('success', 'Question publiée.');
            res.redirect('/ads/' + ad.id);
          })
          .catch((err) => {
            req.flash(
                'error',
                'Une erreur est survenue lors du chargement des données.',
            );
            res.redirect(`/ads/${req.params.id}`);
          });
    })
    .post(
        '/:id/questions/:questionId/answers/create',
        authorize('client'),
        function(req, res) {
          const answer = {
            author: req.user.name,
            body: req.body.body,
            date: new Date(),
          };

          adModel.Ad.findByIdAndUpdate(
              req.params.id,
              {
                $push: {
                  'questions.$[target].answers': answer,
                },
              },
              {
                arrayFilters: [{'target._id': req.params.questionId}],
              },
          )
              .then((ad) => {
                req.flash('success', 'Réponse publiée.');
                res.redirect(`/ads/${ad.id}`);
              })
              .catch((err) => {
                console.error(err);
                req.flash(
                    'error',
                    'Une erreur est survenue lors du chargement des données.',
                );
                res.redirect(`/ads/${req.params.id}`);
              });
        },
    )
    .get('/:id/picture/:index', function(req, res, next) {
      adModel.Ad.findOne({_id: req.params.id})
          .then((ad) => {
            if (!ad || !ad.pictures[req.params.index]) {
              throw new Error('Annonce ou image inexistante');
            }
            res.type('image/*');
            res.send(ad.pictures[parseInt(req.params.index)].body);
          })
          .catch(next);
    })
    .get('/draft/:id', authorize('agent'), renderAdAction(false))
    .get('/:id', renderAdAction(true))
    .get('/', function(req, res) {
      adModel.Ad.find({}, {pictures: 0, questions: 0})
          .then((ads) => {
            const publishedAds = ads
                .filter((ad) => ad.published)
                .sort((a, b) => a.title.localeCompare(b.title));

            const notPublishedAds = ads
                .filter((ad) => !ad.published)
                .sort((a, b) => a.title.localeCompare(b.title));

            // on trie les annonces par ordre alphabétique
            res.render('ads/index', {
              title: 'Annonces',
              ads: {
                published: publishedAds,
                notPublished: notPublishedAds,
              },
            });
          });
    });

/**
 * Renders a single ad.
 * @param {*} published
 *    if the ad to find must be published.
 * @return {Function} ad serving route function.
 */
function renderAdAction(published) {
  return function(req, res) {
    adModel.Ad.findOne(
        {_id: req.params.id, published},
        {'pictures.body': 0},
    )
        .then((ad) => {
          if (ad) {
            res.render('ads/show', {title: ad.title, ad});
          } else {
            req.flash('error', 'L\'annonce demandé n\'a pas été trouvé');
            res.redirect('/');
          }
        })
        .catch((err) => {
          req.flash(
              'log',
              'Un problème est survenu lors du chargement des données',
          );
          res.redirect('/');
        });
  };
}

/**
 * Delete a given ad.
 * @param {*} req
 * @param {*} res
 * @param {Function} next
 */
function deleteAdAction(req, res, next) {
  const id = req.params.id;

  adModel.Ad.deleteOne({_id: id})
      .then((value) => {
        if (value.n !== 0) {
          req.flash('success', 'L\'annonce a bien été supprimée');
        } else {
          req.flash('info', 'L\'annonce n\'a pas été trouvée');
        }
      })
      .catch((reason) => {
        req.flash('error', 'L\'annonce n\'a pas pu être supprimée');
      })
      .finally(() => res.redirect('/ads/'));
}

module.exports = router;
