const express = require('express');
const multer = require('multer');
const router = new express.Router();
const authorize = require('../src/permissions').authorize;

const adModel = require('../models/ad');

const upload = multer({
  storage: multer.memoryStorage(),
});

/* GET to get to get to the ad creation form */
router
    .get('/create', authorize('agent'), function(req, res) {
      res.render('ads/create');
    })
    .post(
        '/create',
        authorize('agent'),
        upload.array('pictures', 3),
        function(req, res, next) {
          const body = req.body;
          const id = body.id;

          const formData = {
            title: body.title,
            type: body.type,
            transactionStatus: body.transactionStatus,
            price: body.price.replace(',', '.'),
            published: body.published === 'on',
            description: body.description,
            availabilityDate:
          body.availabilityDate === '' ? null : body.availabilityDate,
          };

          if (req.files.length) {
            formData.pictures = req.files.map((f) => ({
              name: f.fieldName,
              body: f.buffer,
            }));
          }

          if (id) {
            // Peut-être charger l'objet en amont et le retourner si erreur ?
            adModel.Ad.updateOne({_id: id}, {$set: formData})
                .exec()
                .then(() => {
                  req.flash('success', 'Mise à jour réussie !');
                  res.redirect('/ads/');
                })
                .catch((reason) => {
                  res.render('ads/create', reason);
                });
          } else {
            const newAd = new adModel.Ad(formData);

            newAd
                .save()
                .then(() => {
                  req.flash('success', 'Mise à jour réussie !');
                  res.redirect('/ads/');
                })
                .catch((reason) => {
                  console.error(reason);
                  res.render('ads/create', reason);
                });
          }
        },
    )
    .get('/update/:id', authorize('agent'), function(req, res, next) {
      const id = req.params.id;

      adModel.Ad.findOne({_id: id}, {'pictures.body': 0, 'questions': 0},
          function(err, ad) {
            const errors = [];
            if (err) {
              errors.push(err.message);
            } else if (!ad) {
              errors.push('L\'annonce cherchée n\'a pas été trouvée');
            } else {
              req.flash('success', 'Mise à jour réussie !');
            }

            res.render('ads/create', {ad: ad, errors_update: errors});
          });
    })
    .get('/delete/:id', authorize('agent'), deleteAdAction)
    .delete('/delete/:id', authorize('agent'), deleteAdAction)
    .get('/:id/picture/:index', function(req, res, next) {
      adModel.Ad.findOne({_id: req.params.id})
          .then((ad) => {
            res.type('image/*');
            res.send(ad.pictures[parseInt(req.params.index)].body);
          })
          .catch(next);
    })
    .get('/draft/:id', authorize('agent'), renderAdAction(false))
    .get('/:id', renderAdAction(true))
    .get('/', function(req, res) {
      adModel.Ad.find({}, {pictures: 0, questions: 0})
          .then((ads) => {
            const publishedAds = ads
                .filter((ad) => ad.published)
                .sort((a, b) => a.title.localeCompare(b.title));

            const notPublishedAds = ads
                .filter((ad) => !ad.published)
                .sort((a, b) => a.title.localeCompare(b.title));

            // on trie les annonces par ordre alphabétique
            res.render('ads/index', {
              ads: {
                published: publishedAds,
                notPublished: notPublishedAds,
              },
            });
          })
          .catch((err) => {
            console.error(err);
            res.status(500);
            res.render('ads/index', {
              ads: [],
              _messages: [
                ...res.locals._messages,
                'Un problème est survenu lors du chargement des données',
              ],
            });
          });
    });

/**
 * Renders a single ad.
 * @param {*} published
 *    if the ad to find must be published.
 * @return {Function} ad serving route function.
 */
function renderAdAction(published) {
  return function(req, res) {
    adModel.Ad.findOne({_id: req.params.id, published}, {'pictures.body': 0})
        .then((ad) => {
          if (ad) {
            res.render('ads/show', {ad});
          } else {
            req.flash('error', 'L\'annonce demandé n\'a pas été trouvé');
            res.status(404);
            res.redirect('/');
          }
        })
        .catch((err) => {
          console.error(err);
          req.flash(
              'error',
              'Un problème est survenu lors du chargement des données',
          );
          res.status(500);
          res.redirect('/');
        });
  };
}

/**
 * Delete a given ad.
 * @param {*} req
 * @param {*} res
 * @param {Function} next
 */
function deleteAdAction(req, res, next) {
  const id = req.params.id;

  adModel.Ad.deleteOne({_id: id})
      .then(() => {
        req.flash('success', 'L\'annonce a bien été supprimée');
      })
      .catch((reason) => {
        req.flash('error', 'L\'annonce n\'a pas pu être supprimée');
      })
      .finally(() => res.redirect(303, '/ads/'));
}

module.exports = router;
