const adModel = require('../models/ad');
const userModel = require('../models/user');

const chai = require('chai');
const chaiHttp = require('chai-http');

// mocha needs it
// eslint-disable-next-line no-unused-vars
const should = chai.should();

process.env.NODE_ENV = 'test';
require('dotenv').config();
const server = require('../app');

chai.use(chaiHttp);

let cookie;

describe('Les annonces', () => {
  const agent = chai.request.agent(server);
  const agentData = {
    name: 'agent',
    role: 'agent',
    email: 'test@agent.test.fr',
    password: 'Test123!',
  };

  before(async () => {
    await adModel.Ad.deleteMany({}, (err) => {
      if (err) {
        console.error(err);
      }
    }).exec();

    await userModel.User.deleteMany({}, (err) => {
      if (err) {
        console.error(err);
      }
    }).exec();

    await agent
        .post('/register')
        .send(agentData);

    const cookies = await agent
        .post('/login')
        .send({
          username: agentData.email,
          password: agentData.password,
        }).cookies;

    cookie = cookies.split('=')[1];
  });

  // Requête de consultation des annonces
  describe('Action de listing des annonces', () => {
    it('Supposément obtient les annonces décrites ci-dessous', (done) => {
      agent
          .get('/ads/')
          .set('Cookie', cookie)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(200);
            done();
          });
    });
  });

  // Requêtes de création d'annonce
  describe('Actions de création des annonces', () => {
    it('Supposément obtient la page de création d\'annonce', (done) => {
      agent
          .get('/ads/create')
          .set('Cookie', cookie)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(200);
            done();
          });
    });

    it('Supposément crée une annonce correcte', (done) => {
      const formData = {
        '_method': 'put',
        'title': 'Test',
        'type': adModel.AD_TYPE.VALUE.RENTAL,
        'transactionStatus': adModel.AD_TRANSACTIONSTATUS.VALUE.AVAILABLE,
        'price': '420,69',
        'published': undefined,
      };

      agent
          .post('/ads/create')
          .type('form')
          .send(formData)
          .end((err, res) => {
            res.should.have.status(200);
            done();
          });
    });

    it('Supposément rejette l\'annonce ' +
      'car il lui manque des champs requis', (done) => {
      const formData = {
        '_method': 'put',
        'title': 'Manque des champs obligatoires',
      };

      agent
          .post('/ads/create')
          .type('form')
          .send(formData)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(406);
            done();
          });
    });

    it('Supposément rejette l\'annonce ' +
      'car le champ price pas conforme', (done) => {
      const formData = {
        '_method': 'put',
        'title': 'Champ price pas conforme',
        'type': adModel.AD_TYPE.VALUE.RENTAL,
        'transactionStatus': adModel.AD_TRANSACTIONSTATUS.VALUE.AVAILABLE,
        'price': '99999,969',
        'published': 'on',
      };

      agent
          .post('/ads/create')
          .type('form')
          .send(formData)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(406);
            done();
          });
    });

    it('Supposément rejette l\'annonce ' +
      'car le champ price pas conforme', (done) => {
      const formData = {
        '_method': 'put',
        'title': 'Champ price pas conforme',
        'type': adModel.AD_TYPE.VALUE.RENTAL,
        'transactionStatus': adModel.AD_TRANSACTIONSTATUS.VALUE.AVAILABLE,
        'price': '99999,969',
        'published': 'on',
      };

      agent
          .post('/ads/create')
          .type('form')
          .send(formData)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(406);
            done();
          });
    });
  });

  // Requêtes de mise à jour d'annonce
  describe('Action de mise à jour d\'annonce', (done) => {
    it('Supposément me donne la page vu que l\'id existe', (done) => {
      adModel.Ad.findOne({}).exec()
          .then((value) => {
            agent
                .get(`/ads/update/${value.id}`)
                .end((err, res) => {
                  res.should.be.html;
                  res.should.have.status(200);
                });
          })
          .catch((reason) => {
            console.log(reason);
            chai.assert(true);
          })
          .finally(() => done());
    });

    it('Supposément plante vu que l\'id existe pas en fait', (done) => {
      agent
          .get(`/ads/update/5`)
          .end((err, res) => {
            res.should.be.html;
            res.should.have.status(404);
            done();
          });
    });
  });

  // Requêtes de suppression d'annonce
  describe('Action de suppression d\'annonce', (done) => {
    it('Supposément plante vu que l\'id existe pas en fait', (done) => {
      agent
          .delete(`/ads/delete/aaaaaaaaaaaaaaaaaaaaaaaa`)
          .end((err, res) => {
            res.should.have.status(404);
            done();
          });
    });

    it('Supposément supprime bien vu que l\'id existe', (done) => {
      adModel.Ad.findOne({}).exec()
          .then((value) => {
            agent
                .delete(`/ads/delete/${value.id}`)
                .end((err, res) => {
                  res.should.be.html;
                  res.should.have.status(200);
                });
          })
          .catch((reason) => {
            console.log(reason);
            chai.assert(true);
          })
          .finally(() => done());
    });

    it('Supposément part en couille ' +
    'vu que l\'id est pas sous le bon format', (done) => {
      agent
          .delete(`/ads/delete/5`)
          .end((err, res) => {
            res.should.have.status(500);
            done();
          });
    });
  });
});
