<?php

namespace App\DataFixtures;

use App\Entity\TaxData;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;

class AppFixtures extends Fixture
{
    private const CSV_FILE = '/app/resources/rei.csv';
    private const BATCH_SIZE = 500;

    public function load(ObjectManager $manager): void
    {
        if (!file_exists(self::CSV_FILE)) {
            echo "ERREUR : Fichier introuvable.\n";
            return;
        }

        ini_set('memory_limit', '512M');
        $handle = fopen(self::CSV_FILE, 'r');
        
        // 1. Détection du séparateur
        $headerLine = fgets($handle);
        // CORRECTION ICI : strpos et non str_pos
        $separator = (strpos($headerLine, ';') !== false) ? ';' : ',';
        rewind($handle);

        // 2. Lecture des en-têtes
        // On force l'encodage UTF-8 si le fichier vient d'Excel Windows (Windows-1252)
        // (C'est une astuce pour éviter les problèmes d'accents)
        $header = fgetcsv($handle, 0, $separator);
        
        // Nettoyage des BOM (caractères invisibles de début de fichier)
        $header[0] = preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $header[0]);

        $headerMap = array_flip($header);

        // --- MAPPING (Copie exacte de tes colonnes) ---
        $colRegion      = 'LIBELLE REGION';
        $colRegionCode  = 'CODE REGION';
        $colDept        = 'DEPARTEMENT';
        $colCommune     = 'COMMUNE'; 
        
        $colTauxTFPNB = 'TAFNB - COMMUNE / TAUX NET';
        $colTauxTFPB  = 'FB - COMMUNE / TAUX NET';
        $colTauxTH    = 'TH - COMMUNE / TAUX NET';
        $colTauxCFE   = 'CFE - COMMUNE /TAUX NET'; // Attention, pas d'espace après le / ici selon ton log

        // Colonnes Volumes
        $colVolTFPNB = 'TAFNB - COMMUNE / MONTANT REEL NET';
        $colVolTFPB  = 'FB - COMMUNE / MONTANT REEL';
        $colVolTH    = 'TH - COMMUNE / MONTANT REEL COMMUNAL AU PROFIT DE LA COMMUNE';
        $colVolCFE   = 'CFE - COMMUNE / PRODUIT REEL NET';

        // --- DIAGNOSTIC EN CAS D'ERREUR ---
        if (!isset($headerMap[$colTauxTFPNB])) {
            echo "\n⚠️  ERREUR DE MAPPING : La colonne '$colTauxTFPNB' est introuvable.\n";
            echo "Voici les 10 premières colonnes que je vois dans le fichier :\n";
            print_r(array_slice($header, 0, 10));
            echo "...\n";
            return;
        }

        echo "Début de l'import (Séparateur : '$separator')...\n";
        $i = 0;

        while (($row = fgetcsv($handle, 0, $separator)) !== false) {
            if (count($row) < 5) continue;

            $taxData = new TaxData();
            $taxData->setYear(2023);

            // Géographie
            $taxData->setRegionName($row[$headerMap[$colRegion]] ?? 'Inconnu');
            $taxData->setRegionCode($row[$headerMap[$colRegionCode]] ?? '00');
            $taxData->setDepartmentCode($row[$headerMap[$colDept]] ?? '');
            
            // Code Commune (gestion colonne Code INSEE si dispo, sinon concaténation)
            if (isset($headerMap['code INSEE'])) {
                $taxData->setCommuneCode($row[$headerMap['code INSEE']]);
            } else {
                 $dept = $row[$headerMap[$colDept]] ?? '';
                 $com  = $row[$headerMap[$colCommune]] ?? '';
                 $taxData->setCommuneCode($dept . $com);
            }

            // Nettoyage float
            $cleanFloat = function($val) {
                if ($val === '' || $val === null) return null;
                $val = str_replace([' ', "\xc2\xa0"], '', $val); // Enlève espaces
                $val = str_replace(',', '.', $val);
                return (float) $val;
            };

            // Taux
            $taxData->setRateTFPNB($cleanFloat($row[$headerMap[$colTauxTFPNB]] ?? null));
            $taxData->setRateTFPB($cleanFloat($row[$headerMap[$colTauxTFPB]] ?? null));
            $taxData->setRateTH($cleanFloat($row[$headerMap[$colTauxTH]] ?? null));
            $taxData->setRateCFE($cleanFloat($row[$headerMap[$colTauxCFE]] ?? null));

            // Volumes
            $vol = $cleanFloat($row[$headerMap[$colVolTFPNB]] ?? 0)
                 + $cleanFloat($row[$headerMap[$colVolTFPB]] ?? 0)
                 + $cleanFloat($row[$headerMap[$colVolTH]] ?? 0)
                 + $cleanFloat($row[$headerMap[$colVolCFE]] ?? 0);
            
            $taxData->setCollectedVolume($vol);

            $manager->persist($taxData);

            $i++;
            if ($i % self::BATCH_SIZE === 0) {
                $manager->flush();
                $manager->clear();
                echo "Ligne $i traitée...\r";
            }
        }

        $manager->flush();
        $manager->clear();
        fclose($handle);

        echo "\n✅ IMPORT TERMINÉ : $i lignes insérées.\n";
    }
}