import React, { useMemo, useRef } from "react";
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend,
  ChartOptions,
} from "chart.js";
import { Line } from "react-chartjs-2";
import { RegionEvolutionPoint } from "../../../models/RegionEvolution";
import { colorForRegion } from "./timeseries.utils";

ChartJS.register(
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  Title,
  Tooltip,
  Legend
);

type Props = {
  series: Map<string, RegionEvolutionPoint[]>;
  raw: RegionEvolutionPoint[];
  selectedRegion: string | null;
  onSelectRegion: (region: string | null) => void;
};

export const TimeSeriesChart: React.FC<Props> = ({
  series,
  selectedRegion,
  onSelectRegion,
}) => {
  const chartRef = useRef<ChartJS<"line">>(null);

  const data = useMemo(() => {
    const datasets = Array.from(series.entries()).map(([regionName, pts]) => ({
      label: regionName,
      data: pts.map((p) => ({ x: p.year, y: p.rate })),
      borderColor: colorForRegion(regionName),
      backgroundColor: colorForRegion(regionName),
      borderWidth: selectedRegion === regionName ? 3 : 1.5,
      pointRadius: 0,
      pointHoverRadius: 6,
      tension: 0,
    }));

    return { datasets };
  }, [series, selectedRegion]);

  const options: ChartOptions<"line"> = useMemo(
    () => ({
      responsive: true,
      maintainAspectRatio: false,
      interaction: {
        mode: "nearest",
        axis: "x",
        intersect: false,
      },
      plugins: {
        legend: {
          position: "top",
          onClick: (e, legendItem, legend) => {
            const regionName = legendItem.text;
            onSelectRegion(regionName);
          },
        },
        tooltip: {
          callbacks: {
            label: (context) => {
              const regionName = context.dataset.label || "";
              const rate = context.parsed.y?.toFixed(2);
              return `${regionName}: ${rate}`;
            },
            title: (items) => {
              return `Année ${items[0].parsed.x}`;
            },
          },
        },
      },
      scales: {
        x: {
          type: "linear",
          title: {
            display: false,
          },
          ticks: {
            callback: (value) => Math.round(value as number),
          },
        },
        y: {
          title: {
            display: false,
          },
          ticks: {
            callback: (value) => (value as number).toFixed(1),
          },
        },
      },
      onClick: (event, elements) => {
        if (elements.length > 0) {
          const datasetIndex = elements[0].datasetIndex;
          const regionName = data.datasets[datasetIndex].label;
          onSelectRegion(regionName);
        }
      },
    }),
    [onSelectRegion, data]
  );

  return (
    <div className="chart-wrapper" style={{ width: "100%", height: 500 }}>
      <Line ref={chartRef} data={data} options={options} />
    </div>
  );
};
