// @ts-check
const { test, expect } = require("@playwright/test");

/**
 * @param {import("@playwright/test").Locator} selectLocator
 * @param {string[]} candidates
 * @returns {Promise<string>}
 */
async function selectFirstAvailable(selectLocator, candidates) {
  await expect(selectLocator).toBeVisible();
  await expect(selectLocator).toBeEnabled();
  await expect
    .poll(async () => await selectLocator.locator("option").count(), { timeout: 15000 })
    .toBeGreaterThan(0);

  /** @type {string[]} */
  const values = await selectLocator.locator("option").evaluateAll((options) =>
    /** @type {HTMLOptionElement[]} */ (options).map(
      (option) => /** @type {HTMLOptionElement} */ (option).value
    )
  );

  const chosen = candidates.find((candidate) => values.includes(candidate)) ?? values[0];
  if (!chosen) {
    throw new Error("Aucune option disponible dans le select.");
  }

  await selectLocator.selectOption(chosen);
  return chosen;
}

/**
 * @param {import("@playwright/test").Locator} selectLocator
 * @param {string[]} [preferredCandidates=[]]
 * @returns {Promise<string>}
 */
async function selectDifferentValue(selectLocator, preferredCandidates = []) {
  await expect(selectLocator).toBeVisible();
  await expect(selectLocator).toBeEnabled();
  await expect
    .poll(async () => await selectLocator.locator("option").count(), { timeout: 15000 })
    .toBeGreaterThan(0);

  /** @type {string[]} */
  const values = await selectLocator.locator("option").evaluateAll((options) =>
    /** @type {HTMLOptionElement[]} */ (options).map(
      (option) => /** @type {HTMLOptionElement} */ (option).value
    )
  );
  const current = await selectLocator.inputValue();

  const preferred = preferredCandidates.find(
    (candidate) => values.includes(candidate) && candidate !== current
  );
  const fallback = values.find((value) => value !== current) ?? current;
  const chosen = preferred ?? fallback;

  if (!chosen) {
    throw new Error("Aucune option exploitable dans le select.");
  }

  await selectLocator.selectOption(chosen);
  return chosen;
}

/**
 * @param {string} responseUrl
 * @param {string} endpoint
 * @param {Record<string, string | number>} expectedParams
 * @returns {boolean}
 */
function matchesExpectedQuery(responseUrl, endpoint, expectedParams) {
  if (!responseUrl.includes(endpoint)) return false;
  const url = new URL(responseUrl);
  return Object.entries(expectedParams).every(
    ([key, value]) => url.searchParams.get(key) === String(value)
  );
}

test.describe("Dashboard E2E", () => {
  test.beforeEach(async ({ page }) => {
    await page.goto("/");
  });

  test("affiche le dashboard et les sections principales", async ({ page }) => {
    await expect(page.getByRole("heading", { name: /Tableau de Bord/i })).toBeVisible();
    await expect(page.getByRole("heading", { name: /Analyse par Commune/i })).toBeVisible();
    await expect(page.locator(".controls-panel")).toHaveCount(3);
    await expect(page.locator("canvas")).toHaveCount(2);
    await expect(page.locator("svg")).toHaveCount(1);
  });

  test("serie temporelle: change taxe et periode et recoit des donnees", async ({ page }) => {
    /** @type {string[]} */
    const capturedUrls = [];
    page.on("request", (request) => {
      const url = request.url();
      if (url.includes("/stats/region-evolution")) {
        capturedUrls.push(url);
      }
    });

    const firstControls = page.locator(".controls-panel").first();

    await firstControls.getByRole("button", { name: "cfe", exact: true }).click();

    await expect
      .poll(
        () => capturedUrls.find((url) => url.includes("tax=cfe")),
        { timeout: 15000 }
      )
      .toBeTruthy();

    const last = capturedUrls[capturedUrls.length - 1];
    expect(last).toContain("/stats/region-evolution");
    const payload = await page.request.get(last).then((res) => res.json());
    const members = payload.member ?? payload["hydra:member"] ?? [];
    expect(Array.isArray(members)).toBeTruthy();
    expect(members.length).toBeGreaterThan(0);
    await expect(page.locator(".no-data-message")).toHaveCount(0);
  });

  test("nuage de points: change departement et taxe et recoit des points", async ({ page }) => {
    /** @type {string[]} */
    const capturedUrls = [];
    await page.route("**/stats/department-correlation**", async (route) => {
      capturedUrls.push(route.request().url());
      await route.continue();
    });

    const secondControls = page.locator(".controls-panel").nth(1);

    await secondControls.getByRole("button", { name: "th", exact: true }).click();
    await expect(secondControls.getByRole("button", { name: "th", exact: true })).toHaveClass(/active/);

    await expect
      .poll(
        () => capturedUrls.find((url) => url.includes("tax=th")),
        { timeout: 15000 }
      )
      .toBeTruthy();

    const last = capturedUrls[capturedUrls.length - 1];
    expect(last).toContain("/stats/department-correlation");
    const payload = await page.request.get(last).then((res) => res.json());
    const members = payload.member ?? payload["hydra:member"] ?? [];
    expect(Array.isArray(members)).toBeTruthy();
    expect(members.length).toBeGreaterThan(0);
  });

  test("diagramme circulaire: filtre annee et taxe et recoit une repartition", async ({ page }) => {
    const thirdControls = page.locator(".controls-panel").nth(2);
    const pieChart = page.locator(".chart-and-details").nth(2);
    const yearSelect = thirdControls.locator("select").first();

    const year = await selectDifferentValue(yearSelect, ["2022", "2021"]);

    const [response] = await Promise.all([
      page.waitForResponse((response) =>
        response.request().method() === "GET" &&
        matchesExpectedQuery(response.url(), "/stats/regional-distribution", {
          tax: "tfpnb",
          year,
        })
      ),
      thirdControls.getByRole("button", { name: "tfpnb", exact: true }).click(),
    ]);

    expect(response.ok()).toBeTruthy();
    expect(response.url()).toContain("tax=tfpnb");
    expect(response.url()).toContain(`year=${year}`);

    const payload = await response.json();
    const members = payload.member ?? payload["hydra:member"] ?? [];
    expect(Array.isArray(members)).toBeTruthy();
    expect(members.length).toBeGreaterThan(0);
    await expect(pieChart.locator("svg")).toBeVisible();
    await expect(pieChart.locator("svg path").first()).toBeVisible();
    const slicesCount = await pieChart.locator("svg path").count();
    expect(slicesCount).toBeGreaterThan(0);
  });

  test("api region-evolution: retourne vide si startYear > endYear", async ({ page }) => {
    const response = await page.request.get(
      "http://localhost:8000/api/stats/region-evolution?tax=tfpb&startYear=2022&endYear=2019"
    );
    expect(response.ok()).toBeTruthy();
    const payload = await response.json();
    const members = payload.member ?? payload["hydra:member"] ?? [];
    expect(Array.isArray(members)).toBeTruthy();
    expect(members.length).toBe(0);
  });
});
