<?php

namespace App\Tests\Api;

use ApiPlatform\Symfony\Bundle\Test\ApiTestCase;
use Doctrine\DBAL\Connection;

class StatsEndpointsTest extends ApiTestCase
{
    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass();
        static::bootKernel();

        /** @var Connection $connection */
        $connection = static::getContainer()->get(Connection::class);
        $connection->executeStatement('DELETE FROM tax_data');

        // Jeu de donnees minimal pour eviter la dependance aux fixtures lourdes.
        $rows = [
            [2019, '11', 'ILE-DE-FRANCE', '75', '75056', 12.5, 20.0, 11.0, 3.2, 100000, 200000, 300000, 400000, 1000000],
            [2020, '11', 'ILE-DE-FRANCE', '75', '75056', 12.7, 20.4, 11.1, 3.4, 110000, 210000, 310000, 410000, 1040000],
            [2022, '32', 'LES-HAUTS-DE-FRANCE', '59', '59350', 22.1, 31.5, 17.3, 7.8, 120000, 220000, 320000, 420000, 1080000],
        ];

        foreach ($rows as $row) {
            $connection->executeStatement(
                'INSERT INTO tax_data (year, region_code, region_name, department_code, commune_code, rate_tfpnb, rate_tfpb, rate_th, rate_cfe, volume_tfpnb, volume_tfpb, volume_th, volume_cfe, collected_volume)
                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',
                $row
            );
        }
    }

    public function testRegionEvolutionEndpointReturnsData(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/region-evolution?tax=tfpb&startYear=2019&endYear=2022');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);

        $this->assertNotEmpty($members, 'Le endpoint region-evolution doit retourner des données.');
        $this->assertArrayHasKey('year', $members[0]);
        $this->assertArrayHasKey('regionCode', $members[0]);
        $this->assertArrayHasKey('regionName', $members[0]);
        $this->assertArrayHasKey('rate', $members[0]);
        $this->assertSame('11', $members[0]['regionCode']);
        $this->assertSame('ILE-DE-FRANCE', $members[0]['regionName']);
    }

    public function testDepartmentCorrelationEndpointReturnsData(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/department-correlation?tax=tfpb&year=2022&department=59');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);

        $this->assertNotEmpty($members, 'Le endpoint department-correlation doit retourner des données.');
        $this->assertArrayHasKey('communeCode', $members[0]);
        $this->assertArrayHasKey('rate', $members[0]);
        $this->assertArrayHasKey('collectedVolume', $members[0]);
        $this->assertSame('59350', $members[0]['communeCode']);
        $this->assertEquals(220000.0, (float) $members[0]['collectedVolume']);
    }

    public function testRegionalDistributionEndpointReturnsData(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/regional-distribution?year=2022');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);

        $this->assertNotEmpty($members, 'Le endpoint regional-distribution doit retourner des données.');
        $this->assertArrayHasKey('regionCode', $members[0]);
        $this->assertArrayHasKey('regionName', $members[0]);
        $this->assertArrayHasKey('collectedVolume', $members[0]);
        $this->assertSame('32', $members[0]['regionCode']);
        $this->assertEquals(220000.0, (float) $members[0]['collectedVolume']);
    }

    public function testDepartmentCorrelationReturnsEmptyForUnknownDepartment(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/department-correlation?tax=tfpb&year=2022&department=00');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);
        $this->assertSame([], $members);
    }

    public function testRegionEvolutionFallsBackWhenTaxIsInvalid(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/region-evolution?tax=invalid&startYear=2019&endYear=2022');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);
        $this->assertNotEmpty($members);
        $this->assertSame('11', $members[0]['regionCode']);
    }

    public function testRegionEvolutionReturnsEmptyWhenStartYearIsGreaterThanEndYear(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/region-evolution?tax=tfpb&startYear=2022&endYear=2019');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);
        $this->assertSame([], $members);
    }

    public function testRegionEvolutionUsesDefaultParameters(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/region-evolution');

        $this->assertResponseIsSuccessful();
        $data = $response->toArray(false);
        $members = $this->extractMembers($data);

        $this->assertNotEmpty($members);
        // Par defaut: tax=tfpb, startYear=2019, endYear=2022.
        $this->assertSame(2019, (int) $members[0]['year']);
        $this->assertSame('11', $members[0]['regionCode']);
        $this->assertEquals(20.0, (float) $members[0]['rate']);

        $years = array_values(array_unique(array_map(
            static fn (array $item): int => (int) $item['year'],
            $members
        )));
        sort($years);
        $this->assertSame([2019, 2020, 2022], $years);
    }

    public function testDepartmentCorrelationFallsBackWhenTaxIsInvalid(): void
    {
        $fallbackResponse = static::createClient()->request('GET', '/api/stats/department-correlation?tax=invalid&year=2022&department=59');
        $expectedResponse = static::createClient()->request('GET', '/api/stats/department-correlation?tax=tfpb&year=2022&department=59');

        $this->assertSame(200, $fallbackResponse->getStatusCode());
        $this->assertSame(200, $expectedResponse->getStatusCode());
        $fallbackMembers = $this->extractMembers($fallbackResponse->toArray(false));
        $expectedMembers = $this->extractMembers($expectedResponse->toArray(false));
        $this->assertSame($expectedMembers, $fallbackMembers);
    }

    public function testDepartmentCorrelationReturnsEmptyWhenDepartmentIsMissing(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/department-correlation?tax=tfpb&year=2022');

        $this->assertResponseIsSuccessful();
        $members = $this->extractMembers($response->toArray(false));
        $this->assertSame([], $members);
    }

    public function testRegionalDistributionReturnsDifferentVolumesDependingOnTax(): void
    {
        $responseTfpb = static::createClient()->request('GET', '/api/stats/regional-distribution?tax=tfpb&year=2022');
        $responseCfe = static::createClient()->request('GET', '/api/stats/regional-distribution?tax=cfe&year=2022');

        $this->assertSame(200, $responseTfpb->getStatusCode());
        $this->assertSame(200, $responseCfe->getStatusCode());
        $tfpbMembers = $this->extractMembers($responseTfpb->toArray(false));
        $cfeMembers = $this->extractMembers($responseCfe->toArray(false));

        $this->assertNotEmpty($tfpbMembers);
        $this->assertNotEmpty($cfeMembers);
        $this->assertSame('32', $tfpbMembers[0]['regionCode']);
        $this->assertSame('32', $cfeMembers[0]['regionCode']);
        $this->assertEquals(220000.0, (float) $tfpbMembers[0]['collectedVolume']);
        $this->assertEquals(420000.0, (float) $cfeMembers[0]['collectedVolume']);
        $this->assertNotSame(
            (float) $tfpbMembers[0]['collectedVolume'],
            (float) $cfeMembers[0]['collectedVolume']
        );
    }

    public function testRegionalDistributionFallsBackWhenTaxIsInvalid(): void
    {
        $fallbackResponse = static::createClient()->request('GET', '/api/stats/regional-distribution?tax=invalid&year=2022');
        $expectedResponse = static::createClient()->request('GET', '/api/stats/regional-distribution?tax=tfpb&year=2022');

        $this->assertSame(200, $fallbackResponse->getStatusCode());
        $this->assertSame(200, $expectedResponse->getStatusCode());
        $fallbackMembers = $this->extractMembers($fallbackResponse->toArray(false));
        $expectedMembers = $this->extractMembers($expectedResponse->toArray(false));
        $this->assertSame($expectedMembers, $fallbackMembers);
    }

    public function testRegionalDistributionReturnsEmptyForUnknownYear(): void
    {
        $response = static::createClient()->request('GET', '/api/stats/regional-distribution?tax=tfpb&year=1990');

        $this->assertResponseIsSuccessful();
        $members = $this->extractMembers($response->toArray(false));
        $this->assertSame([], $members);
    }

    /**
     * API Platform peut renvoyer "member" ou "hydra:member" selon la config.
     *
     * @param array<string, mixed> $data
     * @return array<int, array<string, mixed>>
     */
    private function extractMembers(array $data): array
    {
        if (isset($data['member']) && is_array($data['member'])) {
            return $data['member'];
        }

        if (isset($data['hydra:member']) && is_array($data['hydra:member'])) {
            return $data['hydra:member'];
        }

        return [];
    }
}
