<?php

namespace App\DataFixtures;

use App\Entity\TaxData;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\Persistence\ObjectManager;

class AppFixtures extends Fixture
{
    // Pattern pour trouver tous les fichiers annuels REI
    private const FILE_PATTERN = '/app/resources/rei_*.csv';
    private const BATCH_SIZE = 500;

    public function load(ObjectManager $manager): void
    {
        // 1. Trouver les fichiers
        $files = glob(self::FILE_PATTERN);

        if (empty($files)) {
            echo "⚠️  AUCUN FICHIER TROUVÉ !\n";
            echo "Assurez-vous d'avoir des fichiers nommés 'rei_YYYY.csv' dans le dossier resources.\n";
            return;
        }
        sort($files);

        // Import multi-annees volumineux: on evite les erreurs memoire.
        ini_set('memory_limit', '2048M');
        
        // --- CORRECTION : On a supprimé la ligne setSQLLogger qui n'existe plus en DBAL 3 ---

        foreach ($files as $filePath) {
            // 2. Extraire l'année du nom de fichier (ex: rei_2022.csv -> 2022)
            if (preg_match('/rei_(\d{4})\.csv/', basename($filePath), $matches)) {
                $year = (int)$matches[1];
                $this->importFile($manager, $filePath, $year);
            } else {
                echo "⚠️  Impossible de lire l'année dans le fichier : " . basename($filePath) . "\n";
            }
        }
    }

    private function importFile(ObjectManager $manager, string $filePath, int $year): void
    {
        echo "\n🚀 Démarrage de l'import pour l'année $year (Fichier : " . basename($filePath) . ")\n";

        $handle = fopen($filePath, 'r');
        
        // Détection séparateur
        $headerLine = fgets($handle);
        $separator = (strpos($headerLine, ';') !== false) ? ';' : ',';
        rewind($handle);

        // Lecture En-têtes + Nettoyage BOM
        $header = fgetcsv($handle, 0, $separator);
        if (!$header) return;
        $header[0] = preg_replace('/[\x00-\x1F\x80-\xFF]/', '', $header[0]);
        $headerMap = array_flip($header);

        // --- MAPPING INTELLIGENT MULTI-ANNÉES ---
        $findCol = function(array $candidates) use ($headerMap) {
            foreach ($candidates as $name) {
                if (isset($headerMap[$name])) return $name;
            }
            return null;
        };

        // 1. Colonnes Géographiques
        $colRegion     = $findCol(['LIBELLE REGION', 'LIB_REG']);
        $colRegionCode = $findCol(['CODE REGION', 'COD_REG']);
        $colDept       = $findCol(['DEPARTEMENT', 'COD_DEP', 'C_DEP']);
        $colCommune    = $findCol(['COMMUNE', 'LIB_COM']);
        $colInsee      = $findCol(['code INSEE', 'COD_INSEE']);

        // 2. Colonnes Taux (Alias selon les années)
        // Pour coller a la definition "taux d'imposition vote", on privilegie TAUX VOTE.
        $colTauxTFPNB  = $findCol(['FNB - COMMUNE / TAUX VOTE', 'FNB - COMMUNE / TAUX NET', 'TAFNB - COMMUNE / TAUX NET', 'TAU_TFPNB']);
        $colTauxTFPB   = $findCol(['FB - COMMUNE / TAUX VOTE', 'FB - COMMUNE / TAUX NET', 'TAU_TFPB']);
        $colTauxTH     = $findCol(['TH - COMMUNE / TAUX VOTE', 'TH - COMMUNE / TAUX NET', 'TAU_TH']);
        $colTauxCFE    = $findCol(['CFE - COMMUNE / TAUX VOTE', 'CFE - COMMUNE /TAUX NET', 'CFE - COMMUNE / TAUX NET', 'TAU_CFE']);

        // 3. Colonnes Volumes
        $colVolTFPNB   = $findCol(['FNB - COMMUNE / MONTANT REEL', 'TAFNB - COMMUNE / MONTANT REEL NET', 'MT_REEL_TFPNB']);
        $colVolTFPB    = $findCol(['FB - COMMUNE / MONTANT REEL', 'MT_REEL_TFPB']);
        $colVolTH      = $findCol(['TH - COMMUNE / MONTANT REEL COMMUNAL AU PROFIT DE LA COMMUNE', 'TH - COMMUNE / MONTANT REEL', 'MT_REEL_TH']);
        $colVolCFE     = $findCol(['CFE - COMMUNE / PRODUIT REEL NET', 'MT_REEL_CFE']);

        if (!$colDept) {
            echo "❌ Erreur Mapping Année $year : Colonne Département introuvable.\n";
            fclose($handle);
            return;
        }

        $i = 0;
        while (($row = fgetcsv($handle, 0, $separator)) !== false) {
            if (count($row) < 5) continue;

            $taxData = new TaxData();
            $taxData->setYear($year);

            // Géo
            $taxData->setRegionName($row[$headerMap[$colRegion]] ?? 'Inconnu');
            $taxData->setRegionCode($row[$headerMap[$colRegionCode]] ?? '00');
            $taxData->setDepartmentCode($row[$headerMap[$colDept]] ?? '');
            
            // Code Commune
            if ($colInsee && isset($row[$headerMap[$colInsee]])) {
                $taxData->setCommuneCode($row[$headerMap[$colInsee]]);
            } else {
                 $dept = $row[$headerMap[$colDept]] ?? '';
                 $com  = $row[$headerMap[$colCommune]] ?? '';
                 $taxData->setCommuneCode($dept . $com);
            }

            // Nettoyage float
            $cleanFloat = function($val) {
                if ($val === '' || $val === null) return null;
                $val = str_replace([' ', "\xc2\xa0", ','], ['', '', '.'], $val);
                return (float) $val;
            };

            // Taux
            $taxData->setRateTFPNB($cleanFloat($row[$headerMap[$colTauxTFPNB] ?? ''] ?? null));
            $taxData->setRateTFPB($cleanFloat($row[$headerMap[$colTauxTFPB] ?? ''] ?? null));
            $taxData->setRateTH($cleanFloat($row[$headerMap[$colTauxTH] ?? ''] ?? null));

            // La CFE peut etre porte par differentes colonnes selon le regime fiscal de l'EPCI.
            $readFirstNonZero = function(array $candidates) use ($headerMap, $row, $cleanFloat): ?float {
                foreach ($candidates as $candidate) {
                    if (!isset($headerMap[$candidate])) {
                        continue;
                    }
                    $value = $cleanFloat($row[$headerMap[$candidate]] ?? null);
                    if ($value !== null && $value > 0) {
                        return $value;
                    }
                }
                return null;
            };

            $rateCfe = $readFirstNonZero([
                'CFE - COMMUNE / TAUX VOTE',
                'CFE - COMMUNE /TAUX NET',
                'CFE - COMMUNE / TAUX NET',
                'CFE - INTERCOMMUNALITE / TAUX VOTE / FP UNIQUE OU EN ZAE',
                'CFE - INTERCOMMUNALITE / TAUX NET / FP UNIQUE OU EN ZAE',
                'CFE - INTERCOMMUNALITE / TAUX VOTE / FP UNIQUE',
                'CFE - INTERCOMMUNALITE / TAUX NET / FP UNIQUE',
                'CFE - INTERCOMMUNALITE / TAUX VOTE / FISCALITE ADDITIONNELLE OU FP DE ZONE (HORS ZONE)',
                'CFE - INTERCOMMUNALITE / TAUX NET / FISCALITE ADDITIONNELLE OU FP DE ZONE (HORS ZONE)',
                'TAU_CFE',
            ]);
            if ($rateCfe === null) {
                $rateCfe = $cleanFloat($row[$headerMap[$colTauxCFE] ?? ''] ?? null);
            }
            $taxData->setRateCFE($rateCfe);

            // Volumes par taxe + volume total
            $volTFPNB = $cleanFloat($row[$headerMap[$colVolTFPNB] ?? ''] ?? 0) ?? 0.0;
            $volTFPB = $cleanFloat($row[$headerMap[$colVolTFPB] ?? ''] ?? 0) ?? 0.0;
            $volTH = $cleanFloat($row[$headerMap[$colVolTH] ?? ''] ?? 0) ?? 0.0;
            $volCFE = $readFirstNonZero([
                'CFE - COMMUNE / PRODUIT REEL NET',
                'CFE - INTERCOMMUNALITE / PRODUIT REEL NET / FP UNIQUE OU EN ZAE',
                'CFE - INTERCOMMUNALITE / PRODUIT REEL NET / FP UNIQUE',
                'CFE - INTERCOMMUNALITE / PRODUIT REEL NET / FISCALITE ADDITIONNELLE OU FP DE ZONE (HORS ZONE)',
                'CFE - INTERCOMMUNALITE / PRODUIT REEL NET / FPZ EN ZAE',
                'CFE - INTERCOMMUNALITE / PRODUIT REEL NET / FPE EN ZONE EOLIENNE',
                'MT_REEL_CFE',
            ]) ?? 0.0;

            $taxData->setVolumeTFPNB($volTFPNB);
            $taxData->setVolumeTFPB($volTFPB);
            $taxData->setVolumeTH($volTH);
            $taxData->setVolumeCFE($volCFE);
            $taxData->setCollectedVolume($volTFPNB + $volTFPB + $volTH + $volCFE);

            $manager->persist($taxData);

            $i++;
            if ($i % self::BATCH_SIZE === 0) {
                $manager->flush();
                $manager->clear(); 
                echo "   -> $i lignes traitées...\r";
            }
        }

        $manager->flush();
        $manager->clear();
        fclose($handle);
        
        echo "\n✅ Terminé pour $year ($i lignes).\n";
    }
}
